#' Create comparison table across groups
#'
#' @param object An object of class "xtvfreg"
#' @param equation Character; "mean" or "variance"
#' @param output Character; "data.frame", "kable", or "gt"
#' @param ... Additional arguments passed to formatting functions
#' 
#' @return A formatted table (type depends on output parameter)
#' @importFrom stats coef vcov
#' @export
#' 
#' @examples
#' # Create small simulated dataset
#' set.seed(456)
#' n_id <- 30
#' n_time <- 4
#' panel_data <- data.frame(
#'   id = factor(rep(1:n_id, each = n_time)),
#'   group = factor(rep(c("A", "B"), length.out = n_id * n_time)),
#'   x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' # Create comparison table
#' comparison_table(result, equation = "mean")
comparison_table <- function(object, equation = "mean", output = "data.frame", ...) {
  
  if (!equation %in% c("mean", "variance")) {
    stop("equation must be either 'mean' or 'variance'")
  }
  
  model_name <- if (equation == "mean") "mean_model" else "var_model"
  
  # Extract coefficients and standard errors for each group
  coef_list <- list()
  se_list <- list()
  
  for (g in names(object$results)) {
    model <- object$results[[g]][[model_name]]
    coef_list[[g]] <- coef(model)
    se_list[[g]] <- sqrt(diag(vcov(model)))
  }
  
  # Get all unique variable names
  all_vars <- unique(unlist(lapply(coef_list, names)))
  
  # Create comparison matrix
  result_df <- data.frame(Variable = all_vars, stringsAsFactors = FALSE)
  
  for (g in names(coef_list)) {
    # Coefficients
    coefs <- coef_list[[g]][all_vars]
    ses <- se_list[[g]][all_vars]
    
    # Format as "coef (se)"
    formatted <- sprintf("%.4f (%.4f)", coefs, ses)
    formatted[is.na(coefs)] <- "-"
    
    result_df[[paste0("Group_", g)]] <- formatted
  }
  
  # Add stars for significance
  for (g in names(coef_list)) {
    model <- object$results[[g]][[model_name]]
    summ <- summary(model)
    pvals <- summ$coefficients[, 4]  # p-value column
    
    coef_col <- paste0("Group_", g)
    
    for (i in seq_along(all_vars)) {
      var <- all_vars[i]
      if (var %in% names(pvals)) {
        p <- pvals[var]
        stars <- if (p < 0.01) "***" else if (p < 0.05) "**" else if (p < 0.10) "*" else ""
        if (stars != "" && result_df[[coef_col]][i] != "-") {
          result_df[[coef_col]][i] <- paste0(result_df[[coef_col]][i], stars)
        }
      }
    }
  }
  
  # Format output
  if (output == "data.frame") {
    return(result_df)
  } else if (output == "kable") {
    if (!requireNamespace("knitr", quietly = TRUE)) {
      stop("Package 'knitr' is required for kable output")
    }
    caption <- paste(toupper(substring(equation, 1, 1)), 
                     substring(equation, 2), "Equation Results")
    return(knitr::kable(result_df, caption = caption, ...))
  } else if (output == "gt") {
    if (!requireNamespace("gt", quietly = TRUE)) {
      stop("Package 'gt' is required for gt output")
    }
    tbl <- gt::gt(result_df)
    tbl <- gt::tab_header(
      tbl,
      title = paste(toupper(substring(equation, 1, 1)), 
                    substring(equation, 2), "Equation Results")
    )
    return(tbl)
  } else {
    stop("output must be 'data.frame', 'kable', or 'gt'")
  }
}


#' Plot coefficient comparison across groups
#'
#' @param object An object of class "xtvfreg"
#' @param equation Character; "mean" or "variance"
#' @param variable Optional; specific variable to plot. If NULL, plots all
#' 
#' @return A ggplot2 object
#' @importFrom stats coef vcov
#' @importFrom rlang .data
#' @export
#' 
#' @examples
#' \donttest{
#' # Requires ggplot2 package
#' if (requireNamespace("ggplot2", quietly = TRUE)) {
#'   set.seed(456)
#'   n_id <- 30
#'   n_time <- 4
#'   panel_data <- data.frame(
#'     id = rep(1:n_id, each = n_time),
#'     group = rep(c("A", "B"), length.out = n_id * n_time),
#'     x = rnorm(n_id * n_time)
#'   )
#'   panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#'   panel_data$d_x <- panel_data$x - panel_data$m_x
#'   panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#'   
#'   result <- xtvfreg(
#'     formula = y ~ 1,
#'     data = panel_data,
#'     group = "group",
#'     panel_id = "id",
#'     mean_vars = c("m_x", "d_x"),
#'     var_vars = "m_x",
#'     verbose = FALSE
#'   )
#'   
#'   # Plot coefficients
#'   plot_coefficients(result, equation = "mean")
#' }
#' }
plot_coefficients <- function(object, equation = "mean", variable = NULL) {
  
  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop("Package 'ggplot2' is required for plotting")
  }
  
  model_name <- if (equation == "mean") "mean_model" else "var_model"
  
  # Extract coefficients and confidence intervals
  plot_data <- data.frame()
  
  for (g in names(object$results)) {
    model <- object$results[[g]][[model_name]]
    coefs <- coef(model)
    ses <- sqrt(diag(vcov(model)))
    
    df_group <- data.frame(
      group = g,
      variable = names(coefs),
      estimate = coefs,
      std_error = ses,
      conf_low = coefs - 1.96 * ses,
      conf_high = coefs + 1.96 * ses,
      stringsAsFactors = FALSE
    )
    
    plot_data <- rbind(plot_data, df_group)
  }
  
  # Filter to specific variable if requested
  if (!is.null(variable)) {
    plot_data <- plot_data[plot_data$variable == variable, ]
    if (nrow(plot_data) == 0) {
      stop("Variable not found in results")
    }
  }
  
  # Remove intercept for better visualization
  plot_data <- plot_data[plot_data$variable != "(Intercept)", ]
  
  # Create plot
  p <- ggplot2::ggplot(plot_data, 
                       ggplot2::aes(x = .data$estimate, y = .data$group, color = .data$group)) +
    ggplot2::geom_point(size = 3) +
    ggplot2::geom_errorbarh(ggplot2::aes(xmin = .data$conf_low, xmax = .data$conf_high),
                            height = 0.2) +
    ggplot2::geom_vline(xintercept = 0, linetype = "dashed", color = "gray50") +
    ggplot2::facet_wrap(~ variable, scales = "free_x") +
    ggplot2::labs(
      title = paste(toupper(substring(equation, 1, 1)), 
                    substring(equation, 2), "Equation Coefficients by Group"),
      x = "Coefficient Estimate (95% CI)",
      y = "Group",
      color = "Group"
    ) +
    ggplot2::theme_minimal() +
    ggplot2::theme(
      legend.position = "bottom",
      panel.grid.minor = ggplot2::element_blank()
    )
  
  return(p)
}


#' Get variance decomposition
#'
#' @param object An object of class "xtvfreg"
#' @param group Optional; specific group to extract. If NULL, returns all groups
#' 
#' @return A data frame with variance decomposition results
#' @export
#' 
#' @examples
#' set.seed(456)
#' n_id <- 30
#' n_time <- 4
#' panel_data <- data.frame(
#'   id = factor(rep(1:n_id, each = n_time)),
#'   group = factor(rep(c("A", "B"), length.out = n_id * n_time)),
#'   x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' # Get variance decomposition
#' get_variance_decomp(result)
get_variance_decomp <- function(object, group = NULL) {
  
  if (!inherits(object, "xtvfreg")) {
    stop("object must be of class 'xtvfreg'")
  }
  
  if (!is.null(group) && !group %in% names(object$variance_decomp)) {
    stop("Group not found in results")
  }
  
  if (!is.null(group)) {
    decomp <- object$variance_decomp[[group]]
    df <- data.frame(
      group = group,
      var_total = decomp$var_total,
      var_fitted = decomp$var_fitted,
      var_heterosced = decomp$var_heterosced,
      prop_mean = decomp$prop_mean,
      prop_var = decomp$prop_var,
      prop_unexplained = decomp$prop_unexplained,
      stringsAsFactors = FALSE
    )
  } else {
    df_list <- lapply(names(object$variance_decomp), function(g) {
      decomp <- object$variance_decomp[[g]]
      data.frame(
        group = g,
        var_total = decomp$var_total,
        var_fitted = decomp$var_fitted,
        var_heterosced = decomp$var_heterosced,
        prop_mean = decomp$prop_mean,
        prop_var = decomp$prop_var,
        prop_unexplained = decomp$prop_unexplained,
        stringsAsFactors = FALSE
      )
    })
    df <- do.call(rbind, df_list)
  }
  
  rownames(df) <- NULL
  return(df)
}


#' Export results to LaTeX or CSV
#'
#' @param object An object of class "xtvfreg"
#' @param file Character; output file name
#' @param format Character; "latex" or "csv"
#' @param equation Character; "mean", "variance", or "both"
#' 
#' @return Invisibly returns NULL; called for side effects
#' @export
#' 
#' @examples
#' \donttest{
#' # Create temporary file
#' set.seed(456)
#' n_id <- 30
#' n_time <- 4
#' panel_data <- data.frame(
#'   id = rep(1:n_id, each = n_time),
#'   group = rep(c("A", "B"), length.out = n_id * n_time),
#'   x = rnorm(n_id * n_time)
#' )
#' panel_data$m_x <- ave(panel_data$x, panel_data$id, FUN = mean)
#' panel_data$d_x <- panel_data$x - panel_data$m_x
#' panel_data$y <- 1 + panel_data$m_x + rnorm(n_id * n_time)
#' 
#' result <- xtvfreg(
#'   formula = y ~ 1,
#'   data = panel_data,
#'   group = "group",
#'   panel_id = "id",
#'   mean_vars = c("m_x", "d_x"),
#'   var_vars = "m_x",
#'   verbose = FALSE
#' )
#' 
#' # Export to CSV
#' temp_file <- tempfile(fileext = ".csv")
#' export_table(result, file = temp_file, format = "csv", equation = "mean")
#' 
#' # Clean up
#' unlink(temp_file)
#' }
export_table <- function(object, file, format = "csv", equation = "both") {
  
  if (!format %in% c("csv", "latex")) {
    stop("format must be 'csv' or 'latex'")
  }
  
  # CSV export
  if (format == "csv") {
    if (equation %in% c("mean", "both")) {
      mean_table <- comparison_table(object, equation = "mean")
      utils::write.csv(mean_table, 
                       file = if (equation == "mean") file else sub("\\..*$", "_mean.csv", file),
                       row.names = FALSE)
    }
    
    if (equation %in% c("variance", "both")) {
      var_table <- comparison_table(object, equation = "variance")
      utils::write.csv(var_table,
                       file = if (equation == "variance") file else sub("\\..*$", "_variance.csv", file),
                       row.names = FALSE)
    }
    
    message("Tables exported successfully to CSV")
    return(invisible(NULL))
  }
  
  # LaTeX export (requires modelsummary)
  if (format == "latex") {
    if (!requireNamespace("modelsummary", quietly = TRUE)) {
      message("Package 'modelsummary' required for LaTeX export. Falling back to CSV.")
      return(export_table(object, file, format = "csv", equation = equation))
    }
    
    if (equation == "mean" || equation == "both") {
      mean_models <- lapply(object$results, function(x) x$mean_model)
      names(mean_models) <- paste0("Group_", names(object$results))
      
      modelsummary::modelsummary(
        mean_models,
        output = if (equation == "mean") file else sub("\\..*$", "_mean.tex", file),
        stars = c('*' = 0.1, '**' = 0.05, '***' = 0.01),
        title = "Mean Equation Results"
      )
    }
    
    if (equation == "variance" || equation == "both") {
      var_models <- lapply(object$results, function(x) x$var_model)
      names(var_models) <- paste0("Group_", names(object$results))
      
      modelsummary::modelsummary(
        var_models,
        output = if (equation == "variance") file else sub("\\..*$", "_variance.tex", file),
        stars = c('*' = 0.1, '**' = 0.05, '***' = 0.01),
        title = "Variance Equation Results"
      )
    }
    
    message("Tables exported successfully to LaTeX")
  }
  
  invisible(NULL)
}
