% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/km_wlr_calculations_helpers.R
\name{KM_diff}
\alias{KM_diff}
\title{Kaplan-Meier Difference Between Groups}
\usage{
KM_diff(
  df,
  tte.name = "tte",
  event.name = "event",
  treat.name = "treat",
  weight.name = NULL,
  at_points = sort(df[[tte.name]]),
  alpha = 0.05,
  seedstart = 8316951,
  draws = 0,
  risk.points = NULL,
  draws.band = 0,
  tau.seq = 0.25,
  qtau = 0.025,
  show_resamples = TRUE,
  modify_tau = FALSE
)
}
\arguments{
\item{df}{Data frame containing survival data.}

\item{tte.name}{Character; name of time-to-event variable in \code{df}.}

\item{event.name}{Character; name of event indicator variable in \code{df} (1=event, 0=censored).}

\item{treat.name}{Character; name of treatment group variable in \code{df} (0=control, 1=treatment).}

\item{weight.name}{Character or NULL; name of weights variable in \code{df}.}

\item{at_points}{Numeric vector; time points for calculation. Default: sorted unique event times.}

\item{alpha}{Numeric; significance level for confidence intervals. Default: 0.05.}

\item{seedstart}{Integer; random seed for reproducibility. Default: 8316951.}

\item{draws}{Integer; number of draws for pointwise variance estimation. Default: 0.}

\item{risk.points}{Numeric vector; time points for risk table display.}

\item{draws.band}{Integer; number of draws for simultaneous confidence bands. Default: 0.}

\item{tau.seq}{Numeric; step size for tau sequence when \code{modify_tau=TRUE}. Default: 0.25.}

\item{qtau}{Numeric; quantile for tau range restriction. Default: 0.025.}

\item{show_resamples}{Logical; whether to plot resampled curves. Default: TRUE.}

\item{modify_tau}{Logical; whether to restrict time range for simultaneous bands. Default: FALSE.}
}
\value{
A list containing:
\describe{
\item{at_points}{Time points used in calculations}
\item{surv0, surv1}{Survival estimates for control and treatment groups}
\item{sig2_surv0, sig2_surv1}{Variance estimates for survival curves}
\item{dhat}{Survival difference (S1 - S0) at each time point}
\item{sig2_dhat}{Variance of survival difference}
\item{lower, upper}{Pointwise confidence limits (1 - alpha/2)}
\item{sb_lower, sb_upper}{Simultaneous band limits (if draws.band > 0)}
\item{c_alpha_band}{Critical value for simultaneous band (if draws.band > 0)}
\item{dhat_star}{Matrix of resampled differences (if draws.band > 0)}
\item{Zdhat_star}{Standardized resampled differences (if draws.band > 0)}
}
}
\description{
Calculates the difference in Kaplan-Meier curves between two groups, with confidence
intervals and optional resampling-based simultaneous confidence bands.
}
\details{
This function computes the difference in Kaplan-Meier survival curves, delta(t) = S_1(t) - S_0(t),

along with variance estimates and confidence intervals.

When \code{draws.band > 0}, simultaneous confidence bands are constructed using
the supremum distribution of the standardized difference process. These bands
maintain the specified coverage probability across all time points simultaneously.

The variance is estimated using Greenwood's formula for unweighted data, or
resampling-based methods when \code{draws > 0}.
}
\note{
Treatment must be coded as 0=control, 1=experimental. Event must be binary (0/1).
}
\section{Confidence Intervals vs Bands}{

\itemize{
\item Pointwise CIs (\code{lower}, \code{upper}): Cover the true difference at each time point with probability 1-alpha
\item Simultaneous bands (\code{sb_lower}, \code{sb_upper}): Cover the entire difference curve with probability 1-alpha
}
}

\examples{
library(survival)
str(veteran)
veteran$treat <- as.numeric(veteran$trt) - 1

# Basic KM difference
result <- KM_diff(
  df = veteran,
  tte.name = "time",
  event.name = "status",
  treat.name = "treat"
)

# Plot the difference
plot(result$at_points, result$dhat, type = "s",
     xlab = "Time", ylab = "Survival Difference")

# With simultaneous confidence bands
result_band <- KM_diff(
  df = veteran,
  tte.name = "time",
  event.name = "status",
  treat.name = "treat",
  draws.band = 1000,
  modify_tau = TRUE
)

}
\seealso{
\code{\link{df_counting}} for full survival analysis
\code{\link{plotKM.band_subgroups}} for visualization
\code{\link{cumulative_rmst_bands}} for RMST analysis

Other survival_analysis: 
\code{\link{cox_rhogamma}()},
\code{\link{df_counting}()},
\code{\link{wt.rg.S}()}
}
\concept{plotting_functions}
\concept{survival_analysis}
