#'@include funcval.QME.R
NULL
#'Function for fitting QME
#'
#'Function to find QME estimates of the regression coefficients for regression models with truncated response variables.
#'Uses \code{\link{optim}}. Intended to be called through \code{\link{qme}}, not on its own, since \code{qme} also
#'transforms data into the correct form etc.
#'
#'
#'@param formula a symbolic description of the model to be estimated
#'@param mf  the \code{model.frame} containing the variables to be used when fitting the model. \code{\link{qme}} transforms the model frame to the correct form before calling \code{qme.fit}. If \code{qme.fit} is called on its own the model frame needs to be transformed manually.
#'@param point the point of truncation
#'@param direction the direction of truncation
#'@param cv threshold value to be used, number or numeric vector of length 1. (See \code{\link{qme}}, argument \code{cval}, for more information).
#'@param bet starting values to be used by \code{\link{optim}}. Column matrix with p rows.
#'@param ... additional arguments to be passed to \code{\link{optim}} (see the documentation for \code{\link{qme}} for further details).
#'
#'@return
#'A list with components:
#' \item{startcoef }{the starting values of the regression coefficients used by \code{\link{optim}}}
#' \item{coefficients }{the named vector of coefficients}
#' \item{counts }{number of iterations used by \code{\link{optim}}. See the documentation for \code{\link{optim}} for further details}
#' \item{convergence }{from \code{\link{optim}}. An integer code. 0 indicates successful completion. Possible error codes are \cr 1  indicating that the iteration limit maxit had been reached.\cr 10  indicating degeneracy of the Nelder--Mead simplex.}
#' \item{message }{from \code{\link{optim}}. A character string giving any additional information returned by the optimizer, or \code{NULL}.}
#' \item{residuals }{the residuals of the model}
#' \item{df.residual }{the residual degrees of freedom}
#' \item{fitted.values }{the fitted values}
#'
#' @export
#'@importFrom stats model.response model.matrix optim
#'@author Anita Lindmark and Maria Karlsson
#'@seealso \code{\link{qme}}
#'@examples
#'require(utils)
#'
#'##Model frame
#'n <- 1000
#'x <- rnorm(n,0,2)
#'y <- 2+x+4*rnorm(n)
#'
#'d <- data.frame(y=y, x=x)
#'dl0 <- subset(d, y>0)
#'mf <- model.frame(y~x, data=dl0)
#'
#'##Starting values and threshold value
#'lmmod <- lm(data=mf)
#'bet <- lmmod$coef
#'bet <- matrix(bet)
#'cv <- sqrt(deviance(lmmod)/df.residual(lmmod))
#'
#'str(qme. <- qme.fit(y~x,mf,point=0,direction="left",bet,cv))
#'

qme.fit <- function(formula,mf,point,direction,bet,cv,...)
{
  y <- model.response(mf, "numeric")
  y <- matrix(y)
  x <- model.matrix(formula,mf)

  dots <- list(...)

  if (is.null(dots$control)) control <- list(maxit=2000) else control <- dots$control

  z <- optim(par=bet,fn=funcval.QME,control=control,x=x,y=y,cv=cv)

  z$counts <- z$counts[1]
  if(z$counts < 10)
    warning("Convergence reached after very few iterations. This could be because of incorrect \n specifications (point, direction, starting values etc.) in the function call.")

  b <- z$par

  z$residuals <- y-x%*%b

  if(direction=="right")
  {
    z$par <- -z$par
    bet <- -bet
  }

  if(point!=0)
  {
    z$par[1,1] <- z$par[1,1] + point
    bet[1,1] <- bet[1,1] + point
  }

  z$startcoef <- bet
  rownames(z$startcoef) <- c(dimnames(x)[[2]])
  colnames(z$startcoef) <- c("")
  dimnames(z$par) <- dimnames(z$startcoef)
  z$coefficients <- t(z$par)
  z$df.residual <- length(mf[,1])-ncol(x)
  z$fitted.values <- x%*%z$par
  return(z)
}
