% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vec-fourier.R
\name{fourier_vec}
\alias{fourier_vec}
\title{Fourier Series}
\usage{
fourier_vec(x, period, K = 1, type = c("sin", "cos"), scale_factor = NULL)
}
\arguments{
\item{x}{A date, POSIXct, yearmon, yearqtr, or numeric sequence (scaled to difference 1 for \code{period} alignment)
to be converted to a fourier series.}

\item{period}{The number of observations that complete one cycle.}

\item{K}{The fourier term order.}

\item{type}{Either "sin" or "cos" for the appropriate type of fourier term.}

\item{scale_factor}{Scale factor is a calculated value that scales date sequences to numeric sequences.
A user can provide a different value of scale factor to override the date scaling.
Default: NULL (auto-scale).}
}
\value{
A numeric vector
}
\description{
\code{fourier_vec()} calculates a Fourier Series from a date or date-time index.
}
\details{
\strong{Benefits:}

This function is \code{NA} padded by default so it works well with \code{dplyr::mutate()} operations.

\strong{Fourier Series Calculation}

The internal calculation is relatively straightforward:
\verb{fourier(x) = sin(2 * pi * term * x) or cos(2 * pi * term * x)},
where \code{term = K / period}.

\strong{Period Alignment, period}

The \code{period} alignment with the sequence is an essential part of fourier series calculation.
\itemize{
\item \strong{Date, Date-Time, and Zoo (yearqtr and yearmon) Sequences} - Are scaled to unit difference of 1. This happens internally,
so there's nothing you need to do or to worry about. Future time series will be scaled appropriately.
\item \strong{Numeric Sequences} - Are not scaled, which means you should transform them to a unit difference of 1 so that
your x is a sequence that increases by 1. Otherwise your period and fourier order will be incorrectly calculated.
The solution is to just take your sequence and divide by the median difference between values.
}

\strong{Fourier Order, K}

The fourier order is a parameter that increases the frequency. \code{K = 2} doubles the frequency.
It's common in time series analysis to add multiple fourier orders (e.g. 1 through 5) to account for
seasonalities that occur faster than the primary seasonality.

\strong{Type (Sin/Cos)}

The type of the fourier series can be either \code{sin} or \code{cos}. It's common in time series analysis
to add both sin and cos series.
}
\examples{
library(dplyr)

# Set max.print to 50
options_old <- options()$max.print
options(max.print = 50)

date_sequence <- tk_make_timeseries("2016-01-01", "2016-01-31", by = "hour")

# --- VECTOR ---

fourier_vec(date_sequence, period = 7 * 24, K = 1, type = "sin")

# --- MUTATE ---

tibble(date = date_sequence) \%>\%
    # Add cosine series that oscilates at a 7-day period
    mutate(
        C1_7 = fourier_vec(date, period = 7*24, K = 1, type = "cos"),
        C2_7 = fourier_vec(date, period = 7*24, K = 2, type = "cos")
    ) \%>\%
    # Visualize
    tidyr::pivot_longer(cols = contains("_"), names_to = "name", values_to = "value") \%>\%
    plot_time_series(
        date, value, .color_var = name,
        .smooth = FALSE,
        .interactive = FALSE,
        .title = "7-Day Fourier Terms"
    )

options(max.print = options_old)

}
\seealso{
Fourier Modeling Functions:
\itemize{
\item \code{\link[=step_fourier]{step_fourier()}} - Recipe for \code{tidymodels} workflow
\item \code{\link[=tk_augment_fourier]{tk_augment_fourier()}} - Adds many fourier series to a \code{data.frame} (\code{tibble})
}

Additional Vector Functions:
\itemize{
\item Fourier Series: \code{\link[=fourier_vec]{fourier_vec()}}
\item Box Cox Transformation: \code{\link[=box_cox_vec]{box_cox_vec()}}
\item Lag Transformation: \code{\link[=lag_vec]{lag_vec()}}
\item Differencing Transformation: \code{\link[=diff_vec]{diff_vec()}}
\item Rolling Window Transformation: \code{\link[=slidify_vec]{slidify_vec()}}
\item Loess Smoothing Transformation: \code{\link[=smooth_vec]{smooth_vec()}}
\item Missing Value Imputation for Time Series: \code{\link[=ts_impute_vec]{ts_impute_vec()}}, \code{\link[=ts_clean_vec]{ts_clean_vec()}}
}
}
