% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gentlg.R
\name{gentlg}
\alias{gentlg}
\title{Output a \code{tidytlg} table}
\usage{
gentlg(
  huxme = NULL,
  tlf = "Table",
  format = "rtf",
  colspan = NULL,
  idvars = NULL,
  plotnames = NULL,
  plotwidth = NULL,
  plotheight = NULL,
  wcol = 0.45,
  orientation = "portrait",
  opath = ".",
  title_file = NULL,
  file = NULL,
  title = NULL,
  footers = NULL,
  print.hux = TRUE,
  watermark = NULL,
  colheader = NULL,
  pagenum = FALSE,
  bottom_borders = "old_format",
  border_fns = list(),
  alignments = list()
)
}
\arguments{
\item{huxme}{(optional) For tables and listings, A list of input dataframes
containing all columns of interest. For graphs, either \code{NULL} or
a list of \code{ggplot} objects. Vectorized.}

\item{tlf}{(optional) String, representing the output choice. Choices are
\code{"Table"} \code{"Listing"} \code{"Figure"}. Abbreviations are allowed e.g. \code{"T"} for Table.
Strings can be either upper- or lowercase. Vectorized. (Default = \code{"Table"})}

\item{format}{(optional) String, representing the output format. Choices are
\code{"rtf"} and \code{"html"}. Strings can be either upper- or lowercase.(Default = \code{"rtf"})}

\item{colspan}{(optional) A list of character vectors representing the
spanning headers to be used for the table or listing. The first vector
represents the top spanning header, etc. Each vector should have a length
equal to the number of columns in the output data frame. A spanning header
is identified through the use of the same column name in adjacent elements.
Vectorized.}

\item{idvars}{(optional) Character vector defining the columns of a listing
where repeated values should be removed recursively. If \code{NULL} then
all column names are used in the algorithm. If \code{NA}, then the listing remains
as is.}

\item{plotnames}{(optional) Character vector containing the names of the PNG
files, with their extension to be incorporated for figure outputs.
The PNG files need to be located in the path defined by the
parameter \code{opath}.}

\item{plotwidth}{(optional) Numerical value that indicates the plot width in
cm for figure outputs. (Default = 6)}

\item{plotheight}{(optional) Numerical value that indicates the plot height
in cm for figure outputs. (Default = 5)}

\item{wcol}{(optional) Can be a single numerical value that represents the
width of the first column or a vector, specifying the lengths of all columns
in the final table or listing.\cr
When a single numerical value is used, this will be taken as the column width
for the first column. The other columns will be equally spaced across the
remainder of the available space. Alternatively, a vector can be used to
represent the widths of all columns in the final output. The order of the
arguments needs to correspond to the order of the columns in the \code{huxme}
dataset, that are not part of the formatting algorithms
(e.g. \code{anbr}, \code{roworder}, \code{newpage}, \code{newrow},
\code{indentme}, \code{boldme}, \code{by_value}, \code{by_order}).
The sum of the widths in the vector needs to be less or equal to one. When
\code{format="HTML"} \code{wcol} can take only one value, the width
of the first column. (Default = 0.45).}

\item{orientation}{(optional) String: "portrait" or "landscape".
(Default = "portrait")}

\item{opath}{(optional) File path pointing to the output files
(including PNG files for graphs). (Default = ".").}

\item{title_file}{An Excel file that will be read in
with \code{readxl::read_excel()} to be used as the \code{title} and \code{footers} argument.
The use of \code{title} or \code{footers} will override the values passed by this
argument. The file should be either an \code{xls} or \code{xlsx} file with the columns
\verb{TABLE ID}, \code{IDENTIFIER}, and \code{TEXT}. The file will be read in, subset to
where the \code{tblid} matches the \code{tlf} argument, and identifiers with 'title' or
'footnote' will be used to populate the table.}

\item{file}{(required) String. Output identifier.
File name will be adjusted to be lowercase and have \code{-} and \verb{_} removed,
this will not affect table title.}

\item{title}{(required) String. Title of the output. Vectorized.}

\item{footers}{(optional) Character vector, containing strings of footnotes
to be included. Vectorized.}

\item{print.hux}{(optional) Logical, indicating whether the output should be
printed to RTF \code{('format' = "rtf")} or displayed
as HTML \code{('format' = "HTML")}.
(Default = \code{TRUE}). Note that RTF is written using \code{quick_rtf_jnj()}
function and that the HTML is displayed via the \code{huxtable::print_html}
function.}

\item{watermark}{(optional) String containing the desired watermark for
RTF outputs. Vectorized.}

\item{colheader}{(optional) Character vector that contains the column labels
for a table or listing. Default uses the column labels of \code{huxme}.
Vectorized.}

\item{pagenum}{(optional) Logical. When true page numbers are added on the
right side of the footer section in the format page \code{x/y}.
Vectorized. (Default = \code{FALSE}).}

\item{bottom_borders}{(optional) Matrix or \code{"old_format"}.
A matrix indicating where to add the bottom
borders. Vectorized. See \code{\link[=add_bottom_borders]{add_bottom_borders()}} for more information.
If \code{"old_format"}, then borders are added to the \code{colspan} and \code{colheader}
rows. (Default = "old_format").}

\item{border_fns}{(optional) List. A list of functions that transform
the matrix passed to \code{bottom_borders}. Vectorized. See
\code{\link[=add_bottom_borders]{add_bottom_borders()}} for more information.}

\item{alignments}{(optional) List of named lists. Vectorized.
(Default = \code{list()}) Used to specify individual column or cell alignments.
Each named list contains \code{row}, \code{col}, and \code{value}, which are passed to
\code{\link[huxtable:align]{huxtable::set_align()}} to set the alignments.}
}
\value{
A list of formatted \code{huxtables} with desired
properties for output to an RTF or HTML.
}
\description{
Generate and output a \code{huxtable} with desired properties
During this function call, the \code{huxtable} can be written to an RTF or
displayed in HTML. \code{gentlg} is vectorized, see parameter descriptions
to learn for which arguments.
}
\section{\code{Huxme} Details}{

For tables and listings, formatting of the output can be dictated through the
formatting columns
(\code{newrows}, \code{indentme}, \code{boldme}, \code{newpage}), present in the input dataframe.
The final \code{huxtable} will display all columns of the input dataframe, except
any recognized formatting or sorting columns.
For tables, the algorithm uses the column \code{label} as first column.
The remaining columns are treated as summary columns.
For graphs, you can pass a \code{ggplot} object directly
into \code{huxme} and \code{gentlg} will save a PNG with with \code{ggplot2::ggsave()}
and output an RTF.
}

\examples{

final <- data.frame(
  label = c(
    "Overall", "Safety Analysis Set",
    "Any Adverse event{\\\\super a}", "- Serious Adverse Event"
  ),
  Drug_A = c("", "40", "10 (25\%)", "0"),
  Drug_B = c("", "40", "10 (25\%)", "0"),
  anbr = c(1, 2, 3, 4),
  roworder = c(1, 1, 1, 1),
  boldme = c(1, 0, 0, 0),
  newrows = c(0, 0, 1, 0),
  indentme = c(0, 0, 0, 1),
  newpage = c(0, 0, 0, 0)
)

# Produce output in rtf format
gentlg(
  huxme = final,
  wcol = c(0.70, 0.15, 0.15),
  file = "TSFAEX",
  title = "This is Amazing Demonstration 1",
  footers = c(
    "Note: For demonstrative purposes only",
    "{\\\\super a} Subjects are counted once for any given event."
  )
)

# Pass in column headers instead of using variable name
gentlg(
  huxme = final,
  wcol = c(0.70, 0.15, 0.15),
  file = "TSFAEX",
  colheader = c("", "Drug A", "Drug B"),
  title = "This is Amazing Demonstration 1",
  footers = c(
    "Note: For demonstrative purposes only",
    "{\\\\super a} Subjects are counted once for any given event."
  )
)

# Add spanning bottom borders under the cells in the second row
gentlg(
  huxme = final,
  wcol = c(0.70, 0.15, 0.15),
  file = "TSFAEX",
  colheader = c("", "Drug A", "Drug B"),
  title = "This is Amazing Demonstration 1",
  footers = c(
    "Note: For demonstrative purposes only",
    "{\\\\super a} Subjects are counted once for any given event."
  ),
  border_fns = list(spanning_borders(2))
)

# Use a watermark
gentlg(
  huxme = final,
  wcol = c(0.70, 0.15, 0.15),
  file = "TSFAEX",
  colheader = c("", "Drug A", "Drug B"),
  title = "This is Amazing Demonstration 1",
  footers = c(
    "Note: For demonstrative purposes only",
    "{\\\\super a} Subjects are counted once for any given event."
  ),
  watermark = "Confidential"
)

# Set alignments
gentlg(
  huxme = final,
  file = "TSFAEX",
  alignments = list(
    # Align the second column to the left
    list(row = 1:7, col = 2, value = "left"),

    # Align cell "Drug: B" to the right
    list(row = 2, col = 3, value = "right")
  )
)

# Produce output in HTML format
hux <- gentlg(
  huxme = final,
  file = "TSFAEX",
  colheader = c("", "Drug A", "Drug B"),
  title = "This is Amazing Demonstration 1",
  footers = c(
    "Note: For demonstrative purposes only",
    "{\\\\super a} Subjects are counted once for any given event."
  ),
  watermark = "Confidential",
  format = "HTML",
  print.hux = FALSE
)

# Export to HTML page
huxtable::quick_html(hux, file = "TSFAEX.html", open = FALSE)

# clean up.
file.remove("TSFAEX.html", "tsfaex.rtf")
}
\references{
\url{https://github.com/hughjonesd/huxtable}
}
\author{
Steven Haesendonckx \href{mailto:shaesen2@its.jnj.com}{shaesen2@its.jnj.com}

Pelagia Alexandra Papadopoulou \href{mailto:ppapadop@its.jnj.com}{ppapadop@its.jnj.com}
}
