% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spEigenCov.R
\name{spEigenCov}
\alias{spEigenCov}
\title{Covariance Matrix Estimation with Sparse Eigenvectors}
\usage{
spEigenCov(S, q = 1, rho = 0.5, thres = 1e-09)
}
\arguments{
\item{S}{m-by-m sample covariance matrix. It is required that \code{S} is full-rank. Both real and complex matrices are accepted.}

\item{q}{number of sparse eigenvectors.}

\item{rho}{sparsity weight factor. Any nonnegative number (suggested range [0,1]).}

\item{thres}{threshold value. All the entries of the sparse eigenvectors less or equal to \code{thres} are set to 0. The default value is \code{1e-9}.}
}
\value{
A list with the following components:
\item{\code{vectors}  }{m-by-m matrix, columns corresponding to eigenvectors.}
\item{\code{values}  }{m-by-1 vector corresponding to eigenvalues.}
}
\description{
Estimates the covariance matrix with sparse (orthogonal) eigenvectors (in other words, it jointly estimates the sparse eigenvectors and the eigenvalues).
}
\examples{
\dontrun{
library(sparseEigen)
n <- 600  # samples
m <- 500  # dimension
q <- 3  # number of sparse eigenvectors to be estimated
sp_card <- 0.1*m  # sparsity of each eigenvector

# generate covariance matrix with sparse eigenvectors
V <- matrix(0, m, q)
V[cbind(seq(1, q*sp_card), rep(1:q, each = sp_card))] <- 1/sqrt(sp_card)
V <- cbind(V, matrix(rnorm(m*(m-q)), m, m-q))
V <- qr.Q(qr(V))  # orthogonalize eigenvectors
lmd <- c(100*seq(from = q, to = 1), rep(1, m-q))  # generate eigenvalues
R <- V \%*\% diag(lmd) \%*\% t(V)  # covariance matrix

# generate data
X <- MASS::mvrnorm(n, rep(0, m), R)  # random data with underlying sparse structure

# standard and sparse estimation
res_standard <- eigen(cov(X))
res_sparse <- spEigenCov(cov(X), q)

# show inner product between estimated eigenvectors and originals (the closer to 1 the better)
abs(diag(t(res_standard$vectors) \%*\% V[, 1:q]))  #for standard estimated eigenvectors
abs(diag(t(res_sparse$vectors) \%*\% V[, 1:q]))    #for sparse estimated eigenvectors

# show error between estimated and true covariance
norm(cov(X) - R, type = 'F') #for sample covariance matrix
norm(res_sparse$cov - R, type = 'F') #for covariance with sparse eigenvectors
}
}
\references{
K. Benidis, Y. Sun, P. Babu, D. P. Palomar, "Orthogonal Sparse PCA and Covariance Estimation via Procrustes Reformulation,"
\emph{IEEE Transactions on Signal Processing}, vol. 64, no. 23, pp. 6211-6226, Dec. 2016.
}
\author{
Konstantinos Benidis and Daniel P. Palomar
}
