% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_qi.R
\name{sim_qi}
\alias{sim_qi}
\title{Get simulations from a model object}
\usage{
sim_qi(
  mod,
  nsim = 1000,
  newdata,
  original_scale = TRUE,
  return_newdata = FALSE,
  vcov = NULL
)
}
\arguments{
\item{mod}{a model object}

\item{nsim}{number of simulations to be run, defaults to 1,000}

\item{newdata}{A data frame with a hypothetical prediction grid.
If absent, defaults to the model frame.}

\item{original_scale}{logical, defaults to TRUE. If TRUE, the ensuing
simulations are returned on their original scale. If FALSE, the ensuing
simulations are transformed to a more practical/intuitive quantity that is
potentially more intuitive for the user (e.g. a probability for a logistic
regression). This argument is ignored in the context of
simulations on the linear model.}

\item{return_newdata}{logical, defaults to FALSE. If TRUE, the output returns
additional columns corresponding with the inputs provided to \code{newdata}.
This may facilitate easier transformation along with greater clarity as to
what the simulations correspond.}

\item{vcov}{a manual variance-covariance matrix to supply to the simulation
process. Use with caution, and if you did some kind of on-the-fly standard
error adjustment in your regression model to make your results "robust". If
nothing is supplied, the model's default variance-covariance matrix is used.}
}
\value{
\code{sim_qi()} returns a data frame (as a \code{tibble}) with the
quantities of interest and identifying information about the particular
simulation number. For linear models, or simple generalized linear models
where the dependent variable is either "there" or "not there", the quantity
of interest returned is a single column (called \code{y}). For models where the
underlying estimation of the dependent variable is, for lack of a better
term, "multiple" (e.g. ordinal models with the basic proportional odds
assumption), the columns returned correspond with the number of distinct
values of the outcome. For example, an ordinal model where there are five
unique values of the dependent variable will return columns \code{y1}, \code{y2}, \code{y3},
\code{y4}, and \code{y5}.
}
\description{
\code{sim_qi()} is a function to simulate quantities of interest
from a regression model
}
\section{Supported Models}{
\enumerate{
\item Linear models produced by \code{lm()} in base R.
\item Generalized linear models produced by \code{glm()}. Families (links) include:
\itemize{
\item Binomial (logit, probit)
\item Poisson (log)
}
\item Cumulative link models produced by \pkg{ordinal} package.
\itemize{
\item Links: logit, probit
}
}
\subsection{What \code{original_scale} Does in This Function}{

\code{original_scale} defaults to TRUE in this function. When TRUE, the simulated
quantity that's returned is a quantity on its "original scale." Understanding
what exactly that means requires some knowledge about the model in question
and what exactly the model is estimating on your behalf. In the simple linear
model produced by the \code{lm()} function in base R, this is straightforward (and,
thus, this argument does nothing for that model). The quantity returned is
the estimated value of the dependent variable. However, models with some kind
of link function return fitted values on some particular scale that might not
be so user-friendly (e.g. a probit index, or a natural logged odds). However,
that is the "original scale" on which the fitted values are returned. This
summary table may help you better understand what this argument does with
respect to what you want.\tabular{cccc}{
   \emph{Model Function} \tab \emph{Family (Link)} \tab \emph{original_scale = TRUE} \tab \emph{original_scale = FALSE} \cr
   \code{lm()} \tab NA \tab NA (Estimated value of \emph{y}) \tab NA (Estimated value of \emph{y}) \cr
   \code{glm()} \tab binomial(link='logit') \tab natural logged odds of \emph{y} = 1 \tab probability of \emph{y} = 1 \cr
   \code{glm()} \tab binomial(link='probit') \tab probit index of \emph{y} = 1 \tab probability of \emph{y} = 1 \cr
   \code{glm()} \tab poisson(link='log') \tab logged lambda \tab lambda \cr
   \code{clm()} \tab link = 'logit' \tab natural logged odds of \emph{y} = value \emph{j} \tab probability of \emph{y} = value \emph{j} \cr
   \code{clm()} \tab link = 'probit' \tab probit index of \emph{y} = value \emph{j} \tab probability of \emph{y} = value \emph{j} \cr
   \code{logistf()} \tab NA \tab natural logged odds of \emph{y} = 1 \tab probability of \emph{y} = 1 \cr
}


For ordinal models, I recommend setting \code{original_scale} to be FALSE. The
function, underneath the hood, is actually calculating things on the level of
the probability. It's just transforming back to a natural logged odds or a
probit index, if that is what you say you want.
}

\subsection{Other Details}{

Specifying a variable in \code{newdata} with the exact same name as the
dependent variable (e.g. \code{mpg} in the simple example provided in this
documentation file) is necessary for matrix multiplication purposes. The
function will do that for you if you have not done it yourself. I recommend
letting this function do that for you. For matrix multiplication purposes,
this column this function creates will have a default of 0. It does not
(should not?) matter for the simulations.

If nothing is supplied in \code{newdata}, \code{model.frame()} is called and the
simulations are run on the data that inform the model itself. I don't
recommend this, but it works for debugging purposes.

This function builds in an implicit assumption that your dependent variable
in the regression model is not called \code{y}. Nothing about this function will
misbehave (as far as I know) if your dependent variable is called \code{y} in the
model, but it may lead to some confusion in how you interpret the results of
the simulations. The simulated values are always returned as a column called
\code{y}.

Factors (so-called "fixed effects") behave curiously in this function. For now,
this function will politely assume your factors are \emph{all} present in the
\code{newdata} you create (even if you don't want them). Future updates
will try to understand this behavior better. The only loss here is the
efficiency of the simulation procedure, especially if you are not interested
in simulated values of the dependent variable for particular combinations of
the factor variable.
}
}

\examples{

set.seed(8675309)

M1 <- lm(mpg ~ hp + am, mtcars)

newdat <- data.frame(am = c(0,1), hp = 123)

sim_qi(M1, nsim = 100, newdat, return_newdata = TRUE)

}
