% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster.R
\name{clusterTimeseries}
\alias{clusterTimeseries}
\title{Cluster a processed time-series with k-means.}
\usage{
clusterTimeseries(tset, K = 16, iter.max = 1e+05, nstart = 100,
  nui.thresh = -Inf, verb = 1)
}
\arguments{
\item{tset}{a "timeseries" object returned by
\code{\link{processTimeseries}}}

\item{K}{the number of clusters to be calculated, ie. the argument
\code{centers} of \code{\link[stats:kmeans]{kmeans}}, but here
multiple clusterings can be calculated, ie. \code{K} can be an
integer vector. Note that a smaller cluster number is automatically
chosen, if the data doesn't have more then K different values.}

\item{iter.max}{the maximum number of iterations allowed in
\code{\link[stats:kmeans]{kmeans}}}

\item{nstart}{number of randomized initializations of
\code{\link[stats:kmeans]{kmeans}}: "how many random sets should
be chosen?"}

\item{nui.thresh}{threshold correlation of a data point to a
cluster center; if below the data point will be added to
nuisance cluster 0}

\item{verb}{level of verbosity, 0: no output, 1: progress messages}
}
\value{
Returns a list of class "clustering" comprising of a matrix
    of clusterings, lists of cluster centers, cluster-cluster and
    cluster-position similarity matrices (Pearson correlation) used
    by \code{\link{segmentClusters}}, and additional information
    such as a cluster sorting by similarity and cluster colors that
    allow to track clusters in plots. A plot method exists that
    allows to plot clusters aligned to "timeseries" and "segment"
    plots.
}
\description{
Performs \code{\link[stats:kmeans]{kmeans}} clustering of a
time-series object \code{tset} provided by
\code{\link{processTimeseries}}, and calculates cluster-cluster
and cluster-position similarity matrices as required for
\code{\link{segmentClusters}}.
}
\details{
This function performs one or more time-series clustering(s)
using \code{\link[stats:kmeans]{kmeans}}, and the output of
\code{\link{processTimeseries}} as input. It further calculates
cluster centers, cluster-cluster and cluster-position similarity
matrices (Pearson correlation) that will be used by the main function
of this package, \code{\link{segmentClusters}}, to split the cluster
association sequence into segments, and assigns each segment to
the "winning" input cluster.

The argument \code{K} is an integer vector that sets the requested
cluster numbers (argument \code{centers} in
\code{\link[stats:kmeans]{kmeans}}). However, to avoid errors in batch
use, a smaller \code{K} is chosen, if the data contains less then
\code{K} distinct values.

Nuisance Cluster:
values that were removed during time-series processing, such as
rows that only contain 0 or NA values, will be assigned to
the "nuisance cluster" with cluster label "0". Additionally, a minimal
correlation to any cluster center can be specified, argument
\code{nui.thresh}, and positions without any correlation higher
then this, will also be assigned to the "nuisance" cluster.
Resulting "nuisance segments" will not be shown in the results.

Cluster Sorting and Coloring:
additionally the cluster labels in the result object will be sorted by
cluster-cluster similarity (see \code{\link{sortClusters}}) and cluster
colors assigned (see \code{\link{colorClusters}}) for convenient data
inspection with the plot methods available for each data processing
step (see examples).

Note that the function, in conjunction with
\code{\link{processTimeseries}}, can also be used as a stand-alone
tool for time-series clusterings, specifically implementing the
strategy of clustering the Discrete Fourier Transform of periodic
time-series developed by Machne & Murray (2012)
<doi:10.1371/journal.pone.0037906>, and further analyzed in Lehmann
et al. (2013) <doi:10.1186/1471-2105-14-133>, such as transcriptome
data from circadian or yeast respiratory oscillation systems.
}
\examples{
data(primseg436)
## Discrete Fourier Transform of the time-series, 
## see ?processTimeseries for details
tset <- processTimeseries(ts=tsd, na2zero=TRUE, use.fft=TRUE,
                          dft.range=1:7,  dc.trafo="ash", use.snr=TRUE)
## ... and cluster the transformed time-series
cset <- clusterTimeseries(tset)
## plot methods for both returned objects allow aligned plots
par(mfcol=c(3,1))
plot(tset)
plot(cset)
}
\references{
Machne & Murray (2012)
    <doi:10.1371/journal.pone.0037906>, and Lehmann et al. (2013)
    <doi:10.1186/1471-2105-14-133>
}
