% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/correlation.R
\name{cor_spearman}
\alias{cor_spearman}
\title{Spearman's correlation coefficient for multivariate functional data}
\usage{
cor_spearman(mfD, ordering = "MEI")
}
\arguments{
\item{mfD}{a multivariate functional dataset whose Spearman's correlation
coefficient must be computed, in form of multivariate \code{mfData} object.}

\item{ordering}{the ordering relation to use on functional observations,
either \code{"MEI"} for MEI or \code{"MHI"} for MHI (default is
\code{"MEI"}).}
}
\value{
If the original dataset is bivariate, the function returns only the
scalar value of the correlation coefficient for the two components. When
the number of components is L >2, it returns the whole matrix of Spearman's
correlation coefficients for all the components.
}
\description{
This function computes the Spearman's correlation coefficient for a
multivariate functional dataset, with either a Modified Epigraph Index (MEI)
or Modified Hypograph Index (MHI) ranking of univariate elements of data
components.
}
\details{
Given a multivariate functional dataset, with first components \eqn{X^1_1(t),
X^1_2(t), \ldots, X^1_N(t)}, second components \eqn{X^2_1(t), X^2_2(t),
\ldots, X^2_N(t)}, etc., the function exploits either the MEI or MHI to
compute the matrix of Spearman's correlation coefficients. Such matrix is
symmetrical and has ones on the diagonal. The entry (i, j) represents the
Spearman correlation coefficient between curves of component i and j.

See the references for more details.
}
\examples{

#### TOTALLY INDEPENDENT COMPONENTS

N = 2e2
P = 1e3

grid = seq( 0, 1, length.out = P )

# Creating an exponential covariance function to simulate guassian data
Cov = exp_cov_function( grid, alpha = 0.3, beta = 0.4 )

# Simulating (independent) gaussian functional data with given center and
# covariance function
Data_1 = generate_gauss_fdata( N, centerline = sin( 2 * pi * grid ), Cov = Cov )
Data_2 = generate_gauss_fdata( N, centerline = sin( 2 * pi * grid ), Cov = Cov )

# Using the simulated data as (independent) components of a bivariate functional
# dataset
mfD = mfData( grid, list( Data_1, Data_2 ) )

# Correlation approx. zero (components were created independently)
cor_spearman( mfD, ordering = 'MEI' )

# Correlation approx. zero (components were created independently)
cor_spearman( mfD, ordering = 'MHI' )

#### TOTALLY DEPENDENT COMPONENTS

# Nonlinear transform of first component
Data_3 = t( apply( Data_1, 1, exp ) )

# Creating bivariate dataset starting from nonlinearly-dependent components
mfD = mfData( grid, list( Data_1, Data_3 ) )

# Correlation very high (components are nonlinearly dependent)
cor_spearman( mfD, ordering = 'MEI' )

# Correlation very high (components are nonlinearly dependent)
cor_spearman( mfD, ordering = 'MHI' )

}
\references{
Valencia, D., Romo, J. and Lillo, R. (2015). Spearman coefficient for
functions, \emph{Universidad Carlos III de Madrid technical report},
\code{http://EconPapers.repec.org/RePEc:cte:wsrepe:ws133329}.
}
\seealso{
\code{\link{mfData}}, \code{\link{MEI}}, \code{\link{MHI}}
}
