% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrap_spearman_inference.R
\name{BTestSpearman}
\alias{BTestSpearman}
\title{Bootstrap Hypothesis Test on Spearman Correlation Coefficients for Multivariate Functional Data}
\usage{
BTestSpearman(
  mfD1,
  mfD2,
  bootstrap_iterations = 1000,
  ordering = "MEI",
  normtype = "f",
  verbose = FALSE
)
}
\arguments{
\item{mfD1}{is the first functional dataset, specified in form of \code{mfData} object; it must
be compatible with \code{mfD2}.}

\item{mfD2}{is the second functional dataset, specified in form of \code{mfData} object; it must
be compatible with \code{mfD1}.}

\item{bootstrap_iterations}{is the number of bootstrap iterations to be performed.}

\item{ordering}{is the kind of ordering to be used in the computation of Spearman's correlation
coefficient (default is \code{MEI}).}

\item{normtype}{is the norm to be used when comparing the Spearman correlation matrices of the two
functional datasets (default is Frobenius, allowed values are the same as for parameter \code{type} in
the base function \code{norm}).}

\item{verbose}{a boolean flag specifying whether to print the progress of bootstrap iterations or not (default is FALSE).}
}
\value{
The function returns the estimates of the test's p-value and statistics.
}
\description{
This function performs a bootstrap test that checks whether the Spearman correlation structures
(e.g. matrices) of two populations of compatible multivariate functional data are equal or not.
}
\details{
Given a first multivariate functional population, \eqn{X_1^(i), \ldots, X_n^(i)} with \eqn{i=1, \ldots, L},
defined on the grid \eqn{I = t_1, \ldots, t_P}, and a second multivariate functional population,
\eqn{Y_1^(i), \ldots, Y_m^(i)} with \eqn{i=1, \ldots, L}, defined on the same grid \eqn{I}, the
function performs a bootstrap test to check the hypothesis:

\deqn{H_0: R_X = R_Y}
\deqn{H_1: R_X \neq R_Y,}

where R_X and R_Y denote the L x L matrices of Spearman correlation coefficients of the two populations.

\itemize{
\item{The two functional samples must have the same number of components and must be defined over the same
discrete interval \eqn{t_1, \ldots, t_P}.}
\item{The test is performed through a bootstrap argument, so
a number of bootstrap iterations must be specified as well. A high value for this parameter may result
in slow performances of the test (you may consider setting \code{verbose} to \code{TRUE} to get
hints on the process).}}
}
\examples{
set.seed(1)

N  <- 200
P <- 100
L <- 2

grid <- seq(0, 1, length.out = P)

# Creating an exponential covariance function to simulate Gaussian data
Cov <- exp_cov_function(grid, alpha = 0.3, beta = 0.4)

# Simulating two populations of bivariate functional data
#
# The first population has very high correlation between first and second component
centerline_1 <- matrix(
  data = rep(sin(2 * pi * grid)),
  nrow = L,
  ncol = P,
  byrow = TRUE
)
values1 <- generate_gauss_mfdata(
  N = N,
  L = L,
  correlations = 0.9,
  centerline = centerline_1,
  listCov = list(Cov, Cov)
)
mfD1 <- mfData(grid, values1)

# Pointwise estimate
cor_spearman(mfD1)

# The second population has zero correlation between first and second component
centerline_2 <- matrix(
  data = rep(cos(2 * pi * grid)),
  nrow = L,
  ncol = P,
  byrow = TRUE
)
values2 <- generate_gauss_mfdata(
  N = N,
  L = L,
  correlations = 0,
  centerline = centerline_2,
  listCov = list(Cov, Cov)
)
mfD2 <- mfData(grid, values2)

# Pointwise estimate
cor_spearman(mfD2)

# Applying the test
\donttest{
BTestSpearman(mfD1, mfD2)
}
}
\seealso{
\code{\link{BCIntervalSpearman}}, \code{\link{BCIntervalSpearmanMultivariate}}, \code{\link{mfData}}
}
