% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fingerprint.R
\name{get.fingerprint}
\alias{get.fingerprint}
\title{Generate molecular fingerprints}
\usage{
get.fingerprint(
  molecule,
  type = "standard",
  fp.mode = "bit",
  depth = 6,
  size = 1024,
  substructure.pattern = character(),
  circular.type = "ECFP6",
  verbose = FALSE
)
}
\arguments{
\item{molecule}{A \code{jobjRef} object to an \code{IAtomContaine}}

\item{type}{The type of fingerprint. Possible values are:
  \itemize{
\item standard - Considers paths of a given length. The default is
but can be changed. These are hashed fingerprints, with a
default length of 1024
\item extended - Similar to the standard type, but takes rings and
atomic properties into account into account
\item graph - Similar to the standard type by simply considers connectivity
\item hybridization - Similar to the standard type, but only consider hybridization state
\item maccs - The popular 166 bit MACCS keys described by MDL
\item estate - 79 bit fingerprints corresponding to the E-State atom types described by Hall and Kier
\item pubchem - 881 bit fingerprints defined by PubChem
\item kr - 4860 bit fingerprint defined by Klekota and Roth
\item shortestpath - A fingerprint based on the shortest paths between pairs of atoms and takes into account ring systems, charges etc.
\item signature - A feature,count type of fingerprint, similar in nature to circular fingerprints, but based on the signature
descriptor
\item circular - An implementation of the ECFP6 (default) fingerprint. Other circular types can be chosen by modifying the \code{circular.type} parameter.
\item substructure - Fingerprint based on list of SMARTS pattern. By default a set of functional groups is tested.
}}

\item{fp.mode}{The style of fingerprint. Specifying "`bit`" will return a binary fingerprint,
"`raw`" returns the the original representation (usually sequence of integers) and
"`count`" returns the fingerprint as a sequence of counts.}

\item{depth}{The search depth. This argument is ignored for the
`pubchem`, `maccs`, `kr` and `estate` fingerprints}

\item{size}{The final length of the fingerprint.
This argument is ignored for the `pubchem`, `maccs`, `kr`, `signature`, `circular` and
`estate` fingerprints}

\item{substructure.pattern}{List of characters containing the SMARTS pattern to match. If the an empty list is provided (default) than the functional groups substructures (default in CDK) are used.}

\item{circular.type}{Name of the circular fingerprint type that should be computed given as string. Possible values are: 'ECFP0', 'ECFP2', 'ECFP4', 'ECFP6' (default), 'FCFP0', 'FCFP2', 'FCFP4' and 'FCFP6'.}

\item{verbose}{Verbose output if \code{TRUE}}
}
\value{
an S4 object of class \code{\link[fingerprint]{fingerprint-class}} or \code{\link[fingerprint]{featvec-class}},
which can be manipulated with the fingerprint package.
}
\description{
`get.fingerprint` returns a `fingerprint` object representing molecular fingerprint of
the input molecule.
}
\examples{
## get some molecules
smiles <- c('CCC', 'CCN')
mols <- parse.smiles(smiles)

## get a single fingerprint using MACCS (fast)
fp <- get.fingerprint(mols[[1]], type='maccs')

## get MACCS keys for both molecules
fps <- lapply(mols, get.fingerprint, type='maccs')

## get Signature fingerprint
## feature, count fingerprinter
fps <- lapply(mols, get.fingerprint, type='signature', fp.mode='raw')
## get Substructure fingerprint for functional group fragments
fps <- lapply(mols, get.fingerprint, type='substructure')

## get Substructure count fingerprint for user defined fragments
mol1 <- parse.smiles("c1ccccc1CCC")[[1]]
smarts <- c("c1ccccc1", "[CX4H3][#6]", "[CX2]#[CX2]")
fps <- get.fingerprint(mol1, type='substructure', fp.mode='count',
    substructure.pattern=smarts)

## get ECFP0 count fingerprints
mol2 <- parse.smiles("C1=CC=CC(=C1)CCCC2=CC=CC=C2")[[1]]
fps <- get.fingerprint(mol2, type='circular', fp.mode='count', circular.type='ECFP0')
}
\author{
Rajarshi Guha (\email{rajarshi.guha@gmail.com})
}
