% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write_sky_cie.R
\name{write_sky_cie}
\alias{write_sky_cie}
\alias{read_sky_cie}
\title{Write and read out-of-the-box CIE sky model and raster}
\usage{
write_sky_cie(sky_cie, name)

read_sky_cie(name, r, z, a, refit_allowed = FALSE)
}
\arguments{
\item{sky_cie}{list. Object holding the fitted CIE model, diagnostics, and
derived rasters, as produced by the out-of-the-box workflow.}

\item{name}{character vector of length one. File base name without extension.
A path can be included, e.g., \code{"C:/Users/Doe/Documents/DSCN4500"}.}

\item{r}{numeric \link[terra:SpatRaster-class]{terra::SpatRaster} of one layer. The canopy image
used in the out-of-the-box workflow (used by \code{read_sky_cie()} when
refitting).}

\item{z}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=zenith_image]{zenith_image()}}.}

\item{a}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=azimuth_image]{azimuth_image()}}.}

\item{refit_allowed}{logical vector of length one. If \code{TRUE}, allow automatic
re-fit when manual edits are detected.}
}
\value{
See \emph{Functions}.
}
\description{
Persist and restore the out-of-the-box CIE sky model, its diagnostics, and
related rasters/points. The writer produces a human-readable \code{.txt} manifest
plus CSV and GeoPackage sidecar files. The reader reconstructs a list object
compatible with the out-of-the-box pipeline.
}
\details{
Encoding is UTF-8. Decimal point is \code{.}. Unknown keys are
ignored with a warning. Missing required keys trigger an error. The manifest
begins with \verb{format_version:} which is checked for basic compatibility.

When \code{read_sky_cie()} detects manual edits (moved sun disk or changed sky
points) and \code{refit_allowed = TRUE}, it re-fits the CIE model using the
current \code{r}, \code{z}, and \code{a}, then revalidates.
}
\section{Functions}{

\describe{
\item{\code{write_sky_cie}}{No return value. Writes six files to disk with the
prefix \code{name} (see below).}
\item{\code{read_sky_cie}}{Returns a \code{list} similar to the output of
\code{ootb_sky_cie()} and suitable as input to \code{ootb_sky_above()}.}
}
}

\section{Files written by \code{write_sky_cie}}{

\itemize{
\item Plain text manifest: \code{name.txt}
\item CSV with sky radiance samples: \code{name_rr.csv}
\item CSV with sky radiance samples for the upward pass: \code{name_rr_up.csv} (optional)
\item GeoPackage with sky sample points: \code{name_sky_points.gpkg}
\item GeoPackage with the sun disk location: \code{name_sun_disk.gpkg}
\item CSV with validation pairs: \code{name_val.csv}
}
}

\section{Text file keys}{


\describe{
\item{\verb{format_version:}}{Semantic version of the manifest.}
\item{\verb{sun_theta:}}{Solar zenith (deg).}
\item{\verb{sun_phi:}}{Solar azimuth (deg).}
\item{\verb{method_sun:}}{Method used to optimize sun coordinates.}
\item{\verb{zenith_dn:}}{Reference DN at zenith.}
\item{\verb{start_a:}…\verb{start_e:}}{Initial CIE coefficients.}
\item{\verb{is_from_detected_sky_dn:}}{Enables \verb{_rr_up.csv} if \code{TRUE}.}
\item{\verb{fit_a:}…\verb{fit_e:}}{Fitted CIE coefficients.}
\item{\verb{method:}}{Method used to fit CIE coefficients.}
\item{\verb{dist_to_black:}}{Argument passed to \code{extract_sky_points()}.}
\item{\verb{grid:}}{Sky grid parameters (\code{angle_width}, \code{first_ring_different}).}
\item{\verb{min_spherical_dist:}}{Sampling buffer distance (deg).}
\item{\verb{sky_points_no:}}{Number of sky points.}
\item{\verb{outliers_no:}}{Number of sky points that were detected as outliers.}
\item{\verb{rmse:}}{Validation metrics. Root mean squared error.}
\item{\verb{r_squared:}}{Validation metric. Coefficient of determination.}
\item{\verb{mae:}}{Validation metrics. Mean absolute error.}
\item{\verb{[Tested: …]}}{Enumerates tested methods/grids/distances.}
}
}

\examples{
\dontrun{
caim <- read_caim()
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)

# Read a previously written model
path <- system.file("external/example.txt", package = "rcaiman")
sky_cie <- read_sky_cie(gsub(".txt", "", path), r = caim$Blue, z = z, a = a,
                        refit_allowed = TRUE)
}
}
