% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eff_neighbor.R
\name{eff_neighbor}
\alias{eff_neighbor}
\title{Estimation of self and neighbor QTL effects across a genome}
\usage{
eff_neighbor(
  genoprobs,
  pheno,
  smap,
  scale,
  addcovar = NULL,
  addQTL = NULL,
  grouping = rep(1, nrow(smap)),
  response = c("quantitative", "binary"),
  fig = TRUE,
  contrasts = NULL
)
}
\arguments{
\item{genoprobs}{Conditional genotype probabilities as taken from \code{qtl::calc.genoprob()}.}

\item{pheno}{A vector of individual phenotypes.}

\item{smap}{A matrix showing a spatial map for individuals. The first and second column include spatial position along an x-axis and y-axis, respectively.}

\item{scale}{A numeric scalar indicating the maximum spatial distance between a focal individual and neighbors to define neighbor effects.}

\item{addcovar}{An optional matrix including additional non-genetic covariates. It contains no. of individuals x no. of covariates.}

\item{addQTL}{An optional vector containing marker names that are considered covariates. Namely, this option allows composite interval mapping (Jansen 1993).}

\item{grouping}{An optional integer vector assigning each individual to a group. This argument can be used when \code{smap} contains different experimental replicates. Default setting means that all individuals are belong to a single group.}

\item{response}{An optional argument to select trait types. The \code{"quantitative"} or \code{"binary"} calls the \code{"gaussian"} or \code{"binomial"} family in \code{glm()}, respectively.}

\item{fig}{TRUE/FALSE to plot the effects or not.}

\item{contrasts}{An optional vector composed of three TRUE/FALSE values, which represents the presence/absence of specific genotypes as c(TRUE/FALSE, TRUE/FALSE, TRUE/FALSE) = AA, AB, BB. If \code{NULL}, it is compiled from \code{genoprobs} automatically.}
}
\value{
A matrix of estimated additive and dominance deviation for self and neighbor effects, with the chromosome numbers and positions. The row names correspond to marker names.
\itemize{
\item{\code{chr}} {Chromosome number}
\item{\code{pos}} {Marker position}
\item{\code{a1}} {Additive deviation for self effects}
\item{\code{d1}} {Dominance deviation for self effects}
\item{\code{a2}} {Additive deviation for neighbor effects}
\item{\code{d2}} {Dominance deviation for neighbor effects}
}
}
\description{
A function to estimate additive and dominance deviation for self and neighbor QTL effects by a simple regression.
}
\details{
Similar to Haley-Knott regression (Haley & Knott 1992), the additive and dominance deviations are approximated by a regression of trait values on conditional genotype probabilities.
The self QTL effects \code{a1} and \code{d1} are estimated in the same way as the \code{qtl} package performs the Haley-Knott regression.
If \code{contrasts = c(TRUE, TRUE, TRUE)}, neighbor QTL effects \code{a1} and \code{d1} are estimated using a quadratic regression; otherwise, the additive neighbor effects are estimated using a linear regression.
See also Sato, Takeda & Nagano (2021) for the rationale behind the approximation.
}
\examples{
set.seed(1234)
test_map <- qtl::sim.map(len=rep(20,5),n.mar=3,include.x=FALSE)
test_cross <- qtl::sim.cross(test_map,n.ind=50)
test_smap <- cbind(runif(50,1,100),runif(50,1,100))
test_genoprobs <- qtl::calc.genoprob(test_cross,step=2)

test_eff <- eff_neighbor(genoprobs=test_genoprobs,
                         pheno=test_cross$pheno$phenotype,
                         smap=test_smap, scale=20, fig=TRUE
                         )
}
\references{
\itemize{
\item Haley CS, Knott SA (1992) A simple regression method for mapping quantitative trait loci in line crosses using flanking markers. Heredity 69:315-324.
\item Jansen RC (1993) Interval mapping of multiple quantitative trait loci. Genetics 135:205-211.
\item Sato Y, Takeda K, Nagano AJ (2021) Neighbor QTL: an interval mapping method for quantitative trait loci underlying plant neighborhood effects. G3; Genes|Genomes|Genetics 11:jkab017.
}
}
\author{
Yasuhiro Sato (\email{sato.yasuhiro.36c@kyoto-u.jp})
}
