% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getAssortment.R
\name{getAssortment}
\alias{getAssortment}
\alias{getAssortment.data.frame}
\alias{getAssortment.list}
\alias{getAssortment.datBDAT}
\title{Get assortments for one or many trees}
\usage{
getAssortment(tree, ...)

\method{getAssortment}{data.frame}(tree, sort = NULL, mapping = NULL, value = "merge", ...)

\method{getAssortment}{list}(tree, sort = NULL, mapping = NULL, value = "merge", ...)

\method{getAssortment}{datBDAT}(tree, sort = NULL, mapping = NULL, value = "merge", ...)
}
\arguments{
\item{tree}{either an object of class 'datBDAT.assortment' or a data.frame or
list containing the variables needed, i.e. spp, D1, H, and optionally
H1, D2, H2 for specifying the tree. See \code{\link{buildTree}} for more
information and parameter \code{mapping} for mapping of variable names.
Indeed, \code{tree} could additionally take the variables specified in
\code{sort}; otherwise a full outer join is produced between \code{tree}
and \code{sort}}

\item{...}{passing arguments to methods.}

\item{sort}{named list with variables specifying assortments, see
\code{\link{buildTree}}.}

\item{mapping}{mapping of variable names in case a data.frame is given into
parameter \code{tree} between colnames(\code{tree}) and required parameter
names. See details.}

\item{value}{character vector indicating return type: either "Vol", "Skl",
"Fix", "LDSort", "merge" (default) or "raw". See section Value.}
}
\value{
depending on \code{value} either information about 'Skl,' 'Vol' or
'Fix' (these are elements of standard output of Fortran BDAT20 subroutine)
and, additionally, information about the 'Vol' elements is retrieved by
\code{value="LDSort"} or
if \code{value} = 'merge' than a combined information of all produced
assortments with name, base position, assortment length, mid-diameter,
top-diameter and volume is produced. This is most likely what you want,
hence the default. Standard output of BDAT is provided by \code{value="raw"}.
Since v0.4.0 a vectorized BDAT-fortran-function is implemented, so each
element of the returned list in case of value="raw" keeps information of all
trees given. Make sure to appropriately split and manipulate this data.
}
\description{
Function to get assortments given harvest specifications for a
tree of dimension D1, possible D2, and H. Assortments could be diameter
or length constrained.
}
\details{
Parameter 'tree' is able to take either a data.frame with correct
variables names or arbitrary names if \code{mapping} is provided to map the
data.frame names to the required names by \code{c("df-colname" = "var-name")}
or to take a named list.

Assortments are calculated using BDAT2.0 Fortran routines. Slightly extended,
it now is possible to return length and diameter information also about
standard assortments (which are held in "Vol"-element of return list,
if value="raw" or value="Vol").

The standard assortment names are:
\itemize{
  \item X = non-usable wood at stem foot (X-Holz)
  \item Sth = stem wood
  \item Ab = upper part of stem wood, second length after transport cut
  \item Ind = industrial wood
  \item nvDh = non-usable coarse wood
}
}
\section{Methods (by class)}{
\itemize{
\item \code{getAssortment(data.frame)}: transforming \code{data.frame} before calling
\code{getAssortment} using \code{buildTree}

\item \code{getAssortment(list)}: transforming \code{list} before calling
\code{getAssortment} using \code{buildTree}

\item \code{getAssortment(datBDAT)}: class method for class \code{datBDAT}

}}
\examples{
tree <- data.frame(spp = 1, D1 = 30, H = 25)
sort <- list(Az = 7, Sokz = 1)
getAssortment(tree, sort, value = "Vol")
getAssortment(tree, sort, value = "Skl")
sort <- list(Az = 7, Sokz = 1, fixN = 1, fixZ = 10, fixL = 5,
             fixA = 10, fixR = 1)
getAssortment(tree, sort, value = "Vol")
getAssortment(tree, sort, value = "Skl")
getAssortment(tree, sort, value = "Fix")
getAssortment(tree, sort, value = "LDSort")
getAssortment(tree, sort, value = "merge")

## prepare data for repeated sorting
## (get rid of preparating data handling)
n <- 3
tree <- data.frame(
  spp = rep(1, n), D1 = seq(20, 50, length.out = n),
  H = seq(15, 40, length.out = n)
)
sort <- list(lX = 0, Sokz = 1, Az = 7,
             fixN = 2, fixZ = 10, fixL = 5,
             fixA = 10, fixR = 1)
tree <- buildTree(tree = tree, check = "assortment", vars = sort)
getAssortment(tree, value = "Vol")
getAssortment(tree, value = "Skl")
getAssortment(tree, value = "Fix")
getAssortment(tree, value = "LDSort")
getAssortment(tree, value = "merge")

## to get bare BDAT-Output, use value='raw'
# very long list, each element keeping all trees
getAssortment(tree, value="raw")
# bonus: it returns the calling parameters as well

}
\seealso{
\code{\link{BDATVOLABMR}} and \code{\link{BDATVOLABOR}} for a
BDAT-type wrapper and function to keep back-compatability. To get coarse wood
volume over bark (>=7m diameter over bark) BDAT-type functions are
\code{\link{BDATVOLDHMR}} and \code{\link{BDATVOLDHOR}}.
}
