% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_raw_to_contin_tab.R
\name{convert_raw_to_contin_table}
\alias{convert_raw_to_contin_table}
\title{Convert raw AE-Drug incidence data into a contingency table}
\usage{
convert_raw_to_contin_table(
  rawdata,
  Drug_col_name = "DRUG",
  AE_col_name = "AE",
  id_col_name = "CASEID",
  count_col_name = "COUNT",
  aggregated = FALSE,
  create_other_Drug_col = FALSE,
  other_Drug_excludes = NULL,
  other_Drug_colname = "Other_Drug",
  create_other_AE_row = FALSE,
  other_AE_excludes = NULL,
  other_AE_rowname = "Other_AE",
  ...
)
}
\arguments{
\item{rawdata}{a \code{data.frame} or an object that can be converted to a \code{data.frame}.
Must contain 3 columns (i) DRUG: the drug names/labels, (ii) AE: the AE names, and
either (iii) CASEID: case ids corresponding to each combination of AE and DRUG,
(if \code{aggregated} is \code{FALSE}), or (iii') COUNT: the total number of incidences of
each AE and DRUG combination (if \code{aggregated} is \code{TRUE}). If these columns are named
differently in \code{rawdata}, supply the appropriate column names through \code{Drug_col_name},
\code{AE_col_name}, \code{id_col_name} and \code{count_col_name}.}

\item{Drug_col_name, AE_col_name, id_col_name, count_col_name}{Drug, AE, case id and count
column names in \code{rawdata}. Defaults to \code{DRUG}, \code{AE}, \code{CASEID} and \code{COUNT}.}

\item{aggregated}{logical. Has the incidences been already aggregated/summarized into counts
in \code{rawdata}? If \code{TRUE} then then \code{COUNT} column is used to produce the output
contingency table. If \code{FALSE} (default) incidences are first aggregated into counts
before converting to contingency tables.}

\item{create_other_Drug_col}{logical. Add a column in the contingency table for "Other Drugs"? This
column plays the role of a "baseline" group of drugs that typically do not indicate an adverse event
association with the signal of interest. Care should be taken while determining which drugs to
include in this group; See Ding et al (2020) for guidance.}

\item{other_Drug_excludes}{character vector cataloging Drugs that are NOT to be included in the
column for Other Drugs. If NULL (default) then then no Drugs are included in Other Drugs (i.e.,
\code{other_Drug_excludes} contains all Drugs in the raw data). Ignored if
\code{create_other_Drug_col = FALSE}.}

\item{other_Drug_colname}{character. Row name for the "Other Drug" column created. Ignored if
\code{create_other_Drug_col = FALSE}.}

\item{create_other_AE_row}{logical. Add a row in the contingency table for "Other AEs"? This can aid computation
in situations where there are certain AEs of primary interest. See \code{other_AE_excludes} for details on
how to specify the "Other AE" row.}

\item{other_AE_excludes}{character vector cataloging AEs that are NOT to be included in the
row for Other AEs. If NULL (default) then then no AEs are included in Other AEs (i.e.,
\code{other_AE_excludes} contains all AEs in the raw data). Ignored if  \code{create_other_AE_row = FALSE}.}

\item{other_AE_rowname}{character. Row name for the "Other AE" row created. Defaults to "Other AE". Ignored if
\code{create_other_AE_row = FALSE}.}

\item{...}{unused.}
}
\description{
Convert raw AE-Drug incidence data into a contingency table
}
\details{
This is a convenience function that creates a contingency table cataloging counts of
AE-Drug incidences from a raw Drug/AE incidence data frame.
It accepts both raw incidence data (each row is one incidence of a Drug-AE combination,
indexed by case ids) and summarized count data (each row catalogs the total counts of incidences
of each Drug-AE pair). The output is a matrix (contingency table) enumerating total count of cases for
each pair of AE (along the rows) and drug (along the columns) with appropriately
specified row and column names, and can be passed to a pvlrt() call. See the examples for more details.

The output can be fed into \link{pvlrt} or its wrappers as \code{contin_table}
}
\examples{

# convert to contingency table form incidence (non-aggregated) raw data
# AE subset = AEs in statin46
# Durg subset = union of statin46 and gbca drugs
tab1 <- convert_raw_to_contin_table(
  rawdata = faers22q3raw,
  Drug_col_name = "DRUG",
  AE_col_name = "AE",
  id_col_name = "CASEID",
  aggregated = FALSE,
  other_AE_excludes = rownames(statin46),
  other_Drug_excludes = union(colnames(gbca), colnames(statin)),
  create_other_Drug_col = TRUE,
  create_other_AE_row = FALSE
)

# convert to contingency table AFTER aggregating and counting
# the total number of incidences of each (AE, Drug) pair
## Same AE and Drug subsets as before
## aggregation (counting) done using data.table dt[i, j, by] syntax
## uses magrittr \%>\% pipe
tab2 <- data.table::as.data.table(
  faers22q3raw
)[
  ,
  .(COUNT = length(unique(CASEID))),
  by = .(DRUG, AE)
] \%>\%
  convert_raw_to_contin_table(
    Drug_col_name = "DRUG",
    AE_col_name = "AE",
    count_col_name = "COUNT",
    aggregated = TRUE,
    other_AE_excludes = rownames(statin46),
    other_Drug_excludes = union(colnames(gbca), colnames(statin)),
    create_other_Drug_col = TRUE,
    create_other_AE_row = FALSE
  )

all.equal(tab1, tab2)

# use the contingency table produced above in pvlrt()
## 500 bootstrap iterations (nsim) in the example below
## is for quick demonstration only --
## we recommended setting nsim to 10000 (default) or bigger
test1 <- pvlrt(tab1, nsim = 500)

}
\references{
Ding, Y., Markatou, M. and Ball, R., 2020. An evaluation of statistical approaches to postmarketing surveillance. Statistics in Medicine, 39(7), pp.845-874.

Chakraborty, S., Liu, A., Ball, R. and Markatou, M., 2022. On the use of the likelihood ratio test methodology in pharmacovigilance. Statistics in Medicine, 41(27), pp.5395-5420.
}
