# -----------------------------------------------
# Generate a chronological or ordered p-plot from
# a data vector, using a one-sample t-test to
# test the mean of the data vector against a
# reference value of 0.
# -----------------------------------------------
pplot.t.test <- function(data,
                      sort = FALSE,
                      startindex = 2,
                      produce.plot = TRUE,
                      xtitle = "Observation",
                      plottype = 3,
                      alpha = 0.05,
                      alternative = "two.sided",
                      n.sig = TRUE,
                      n.sig.adjust = -1,
                      ylim.p = c(0,1),
                      ylim.es = -1,
                      conf.level = 0.95,
                      pcol.alpha = 125) {
  
  # data: vector of data values
  # sort: Sort data vector? (TRUE or FALSE)
  # startindex: number of observations to start with (minimum 2)
  # produce.plot: Generate plots (TRUE or FALSE); if false, pplot() can
  #               be used generate data frames for p-values and effect sizes
  #               (incl. confidence intervals) that are the basis of the plots
  # plottype: 1 = only p-plot, 2 = only effect size plot, 3 = both plots
  # n.sig: Add n_sig to p-plot? (TRUE or FALSE)
  # n.sig.adjust: -1 = auto, 0 = left-adjusted, 0.5 = centered, 1 = right
  # ylim.p: y limits for p-plot; vector of min, max
  # ylim.es: y limits for effect size plot; either vector of min, max,
  #         or only one value to be used as minimum with maximum being
  #         determined as a function of the data
  # conf.level: confidence level for effect size CIs
  # pcol.alpha: transparency; set to 255 for solid fill
  #
  # Output: A data frame consisting of either one variable (if plottype==1)
  #         showing p-values as a function of observation numbers, or a
  #         data frame with three variables (if plottype==2) with standardized
  #         means and upper and lower confidence interval bounds, or a
  #         data frame with four variables (if plottype==3), holding the
  #         combined data
  
  # Sort data if required
  if (sort==TRUE) {
    data <- sort(data)
  }
  
  # Check validity of start index
  if (is.na(var(data[1:startindex])) | var(data[1:startindex])==0) {
    message("Invalid start index detected (no variance in data), increasing start index.\n")
    while (is.na(var(data[1:startindex])) | var(data[1:startindex])==0) {
      startindex <- startindex + 1
      if (startindex > length(data)) {
        print("Error - insufficient variance in the data to determine valid start index.")  
        stop(call. = TRUE)
      }
    }
    message(paste0("Working start index: ", startindex,"\n"))
  }
  
  # Determine number of plot panels
  nplots <- 1
  if (plottype > 2) {
    nplots <- 2
  }
  
  # Load MBESS package for effect size CIs
  if (!requireNamespace("MBESS", quietly = TRUE)) {
    warning("Package 'MBESS' is not installed but required for computing confidence intervals for effect sizes (for plottype >= 2).")
    stop(call. = TRUE)
  }
  #if (plottype >= 2) {
  #  req.packages <- c("MBESS")
  #  for (ipckg in c(1:length(req.packages))) {
  #    if (!require(req.packages[ipckg],character.only = TRUE)) {
  #      install.packages(req.packages[ipckg], 
  #                       repos = "https://cloud.r-project.org")
  #    }
  #    library(req.packages[ipckg],character.only = TRUE)
  #  }
  #}
  
  # Get p-values and effect sizes
  ps <- matrix(data=NA,nrow=length(data),ncol=1)
  sms <- matrix(data=NA,nrow=length(data),ncol=3)
  for (isbj in startindex:length(data)) {  
    ps[isbj,1] <- t.test(data[1:isbj],alternative=alternative)$p.value
    if (plottype >= 2) {
      smtmp <- MBESS::ci.sm(sm = mean(data[1:isbj])/sd(data[1:isbj]),
                     N = length(data[1:isbj]),
                     conf.level = conf.level)
      sms[isbj,1] <- smtmp$Standardized.Mean
      sms[isbj,2] <- smtmp$Lower.Conf.Limit.Standardized.Mean
      sms[isbj,3] <- smtmp$Upper.Conf.Limit.Standardized.Mean
    }
  }
  
  # Get y limits for effect size plot (rounding up to nearest .5)
  if (plottype >= 2) {
    if (length(ylim.es)==1) {
      ylim.es <- c(ylim.es,ceiling(max(sms[startindex:nrow(sms),3])/0.5)*0.5)
    }
  }
  
  # Define plot color
  #pcol <- rgb(117, 157, 197, maxColorValue  = 255, alpha = 125, names = "rndteal")
  pcol <- rgb(117, 157, 197, maxColorValue  = 255, alpha = pcol.alpha, names = "rndteal")
  
  # Produce plots?
  if (produce.plot) {
    
    # Set plot layout
    if (plottype < 3) {
      #par(mfrow=c(1,1),las = 1)  
    } else {
      oldpar <- par(no.readonly = TRUE)
      on.exit(par(oldpar))
      par(mfrow=c(1,2),las = 1)
    }
    
    # Do p-plot
    if (plottype!=2) {
      plot(ps,pch = 21,col = pcol, type = "o",
           xlab= xtitle, ylab=expression(italic("p")*"-value"),
           ylim = ylim.p)
      points(ps,pch = 16, col = "white")
      points(ps,pch = 16, col = pcol)
      abline(h=alpha)
      
      # Add n.sig
      if (!n.sig) {
        n.sig <- -1
      } else {
        if (ps[length(ps)]<=alpha) {
          n.sig <- -1
          isig <- length(ps)
          while (n.sig < 0) {
            isig <- isig - 1
            if (isig <= 1) {
              n.sig <- 0
            } else if (is.na(ps[isig])) {
              n.sig <- 0
            } else if (ps[isig]>alpha) {
              n.sig <- isig + 1
            }
          }
          if (n.sig > 0) {
            texty <- ylim.p[1] + 0.90*(ylim.p[2]-ylim.p[1])
            abline(v=n.sig)
            if (n.sig.adjust < 0) {
              if (n.sig<0.5*length(ps)) {
                n.sig.adjust <- 0
              } else {
                n.sig.adjust <- 1
              }
            }
            # Place white box if centered adjustment
            if (n.sig.adjust==0.5) {
              text_width <- 3 #strwidth("example text", cex = 1) * 1.1
              if (sort) {
                text_height <- strheight(
                  bquote("n"["o"]*" = "*.(length(ps)-n.sig)), cex = 1) * 0.9
              } else {
                text_height <- strheight(
                  bquote("n"["c"]*" = "*.(length(ps)-n.sig)), cex = 1) * 0.9
              }
              rect(xleft = n.sig - text_width / 2, 
                   ybottom = texty - text_height / 2,
                   xright = n.sig + text_width / 2,
                   ytop = texty + text_height / 2,
                   col = "white", border = NA)
            }
            if (sort) {
              text(n.sig,texty,bquote("n"["o"]*" = "*.(length(ps)-n.sig)),
                   adj=n.sig.adjust)
            } else {
              text(n.sig,texty,bquote("n"["c"]*" = "*.(length(ps)-n.sig)),
                   adj=n.sig.adjust)
            }
          
          }
        }
      }
    }
    
    # Do effect size plot
    if (plottype>1) {
      plot(sms[,1],ylim = ylim.es,las = 1,pch = 21,col = pcol, type = "o",
           xlab= xtitle, ylab=paste0("Effect size & ",  
                                     round(conf.level*100),"% CI"))
      ribbonx <- c(1:nrow(sms))
      polygon(c(ribbonx, rev(ribbonx)), c(sms[,2], rev(sms[,3])),
              col = adjustcolor(pcol, 0.3), border = pcol)
      abline(v=startindex, col = "white")
      abline(v=nrow(sms), col = "white")
      abline(h=0.0)
      points(sms[,1],pch = 16, col = "white")
      points(sms[,1],pch = 16, col = pcol)
      box()
    }
    
  } # Produce plots?
  
  # Output
  if (plottype == 1) {
    outval <- ps
    outval <- as.data.frame(outval)
    names(outval) <- c("p")
  } else if (plottype == 2) { 
    outval <- sms
    outval <- as.data.frame(outval)
    names(outval) <- c("sm","CIlower","CIupper")
  } else if (plottype == 3) { 
    outval <- cbind(ps,sms)
    outval <- as.data.frame(outval)
    names(outval) <- c("p","sm","CIlower","CIupper")
  } else {
    outval <- "Undefined plot type specified!"
  }
  outval <- outval
}
