#' @title Hixson-Crowell Drug Release Kinetic Model
#' @name hixson_crowell_model
#' @description
#' Fits experimental cumulative drug release data to the Hixson-Crowell cube-root model,
#' which describes drug release from systems where dissolution occurs with
#' a change in surface area and particle diameter over time. The model is
#' based on a linear regression of \eqn{(W_0^{1/3} - W_t^{1/3})} versus time.
#'
#' The function assumes that the initial drug amount (\eqn{W_0}) is known
#' and that the remaining drug amount (\eqn{W_t}) can be calculated from
#' cumulative percent drug remaining data. It supports optional grouping variables
#' (e.g., formulation, batch) and optional pH-dependent analysis. The function
#' generates publication-quality plots with experimental data points, fitted
#' Hixson-Crowell straight lines, and annotations for the Hixson-Crowell rate
#' constant (kHC), intercept, coefficient of determination (R^2), and the time
#' required for 50-percent drug release (t50).
#'
#' Model:
#' \deqn{W_0^{1/3} - W_t^{1/3} = k_{HC} * t}
#'
#' where:
#' - \eqn{W_0} is the initial amount of drug
#' - \eqn{W_t} is the remaining amount of drug at time \eqn{t}
#' - \eqn{k_{HC}} is the Hixson-Crowell dissolution rate constant
#'
#' The Hixson-Crowell model is applicable to dosage forms where drug release
#' is governed by erosion or dissolution with a decreasing surface area
#' (e.g., tablets, pellets).
#'
#' @param data A data frame containing experimental drug release data.
#' @param time_col Character string specifying the column name for time (minutes).
#' @param remain_col Character string specifying the column name for cumulative
#'   percent drug remaining.
#' @param group_col Optional character string specifying a grouping variable
#'   (e.g., formulation, batch).
#' @param pH_col Optional character string specifying a column containing pH values.
#' @param plot Logical; if TRUE, generates a plot with fitted Hixson-Crowell lines.
#' @param annotate Logical; if TRUE, annotates the plot with kHC, intercept, R^2,
#'   and t50 (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats lm na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth geom_text labs
#' theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with Hixson-Crowell parameters
#'         (kHC, intercept), coefficient of determination (R^2), and t50 for each group.}
#'   \item{\code{data}}{Processed data used for model fitting and plotting.}
#' }
#' @examples
#' # Example I: Single formulation
#' df <- data.frame(
#'   time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
#'   remain = c(100, 88.6, 79.2, 69.2, 60.2, 42.2, 28.0, 15.2, 6.5)
#' )
#' hixson_crowell_model(
#'   data = df,
#'   time_col = "time",
#'   remain_col = "remain"
#' )
#'
#' # Example II: Two formulations (grouped, not pH-dependent)
#' df2 <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
#'   remain = c(
#'     100, 90, 81, 72, 63, 54, 45, 36, 27, 18,  # Formulation A
#'     100, 92, 84, 76, 68, 60, 52, 44, 36, 28   # Formulation B
#'   ),
#'   formulation = rep(c("Formulation A", "Formulation B"), each = 10)
#' )
#' hixson_crowell_model(
#'   data = df2,
#'   time_col = "time",
#'   remain_col = "remain",
#'   group_col = "formulation"
#' )
#'
#' # Example III: pH-dependent release
#' df_pH <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
#'   remain = c(
#'     100, 90, 80, 70, 60, 50, 40, 30, 20, 10,  # pH 7.4
#'     100, 92, 84, 76, 68, 60, 52, 44, 36, 28   # pH 4.5
#'   ),
#'   pH = rep(c(7.4, 4.5), each = 10)
#' )
#' hixson_crowell_model(
#'   data = df_pH,
#'   time_col = "time",
#'   remain_col = "remain",
#'   pH_col = "pH"
#' )
#'
#' # Example IV: Two formulations under two pH conditions
#' df1 <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
#'   remain = c(
#'     100, 88, 75, 62, 50, 38, 28, 18, 10, 5,   # pH 4.5
#'     100, 90, 78, 65, 52, 40, 30, 20, 12, 6    # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 10)
#' )
#' df2 <- data.frame(
#'   time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 135), 2),
#'   remain = c(
#'     100, 90, 78, 66, 54, 44, 34, 25, 16, 8,   # pH 4.5
#'     100, 92, 80, 68, 56, 44, 34, 24, 15, 7    # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 10)
#' )
#' df_all <- rbind(
#'   cbind(formulation = "Dataset 1", df1),
#'   cbind(formulation = "Dataset 2", df2)
#' )
#' hixson_crowell_model(
#'   data = df_all,
#'   time_col = "time",
#'   remain_col = "remain",
#'   group_col = "formulation",
#'   pH_col = "pH"
#' )
#' @references Hixson, A. W., & Crowell, J. H. (1931) <doi:10.1021/ie50260a018>
#' Dependence of reaction velocity upon surface and agitation. Industrial & Engineering
#' Chemistry, 23(8), 923–931.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "remaining", "Wt_cuberoot", "Wo_minus_Wt", "group",
                         "kHC", "intercept", "R2", "t50", "label", "x_pos", "y_pos",
                         "hjust", "vjust"))

hixson_crowell_model <- function(data,
                                 time_col = "time",
                                 remain_col = "remaining",
                                 group_col = NULL,
                                 pH_col = NULL,
                                 plot = TRUE,
                                 annotate = TRUE,
                                 normalize = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) stop("Package 'ggplot2' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, remain_col, group_col, pH_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "remaining")

  # Normalize % remaining to fraction if requested
  if (normalize) df$remaining <- df$remaining / 100

  # Ensure time >= 0 and remaining > 0
  df <- df[df$time >= 0 & df$remaining > 0, ]

  # Cube-root transform
  df$Wt_cuberoot <- df$remaining^(1/3)

  # -------------------------
  # Grouping
  # -------------------------
  if (!is.null(group_col) && !is.null(pH_col)) {
    df$group <- paste0(df[[group_col]], " | pH ", df[[pH_col]])
  } else if (!is.null(pH_col)) {
    df$group <- paste0("pH ", df[[pH_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }
  df$group <- as.factor(df$group)

  # -------------------------
  # Compute Wo - Wt^(1/3)
  # -------------------------
  df <- do.call(rbind, lapply(split(df, df$group), function(d) {
    Wo <- d$Wt_cuberoot[d$time == min(d$time)][1]
    d$Wo_minus_Wt <- Wo - d$Wt_cuberoot
    d
  }))

  # -------------------------
  # Hixson-Crowell fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fit <- stats::lm(Wo_minus_Wt ~ time, data = d)
    s <- summary(fit)

    kHC <- coef(fit)[2]
    intercept <- coef(fit)[1]
    R2 <- s$r.squared

    # t50 calculation
    target <- if (normalize) (0.5)^(1/3) else (50)^(1/3)
    Wo <- d$Wt_cuberoot[d$time == min(d$time)][1]

    t50 <- if (!is.na(kHC) && kHC != 0) {
      (Wo - target - intercept) / kHC
    } else NA_real_

    data.frame(
      group = unique(d$group),
      kHC = kHC,
      intercept = intercept,
      R2 = R2,
      t50 = t50
    )
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(
      df,
      ggplot2::aes(x = time, y = Wo_minus_Wt, color = group)
    ) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::geom_smooth(method = "lm", se = FALSE, color = "black", linewidth = 1) +
      ggplot2::labs(
        title = "Hixson-Crowell Drug Release Kinetics",
        subtitle = "Hixson-Crowell Cube-Root Model",
        x = "Time (minutes)",
        y = expression(W[0]^(1/3) - W[t]^(1/3)),
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Conditional annotations
    # -------------------------
    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {

      ann <- fit_results
      ann$label <- paste0(
        "kHC = ", signif(ann$kHC, 3), "\n",
        "Intercept = ", round(ann$intercept, 3), "\n",
        "R^2 = ", round(ann$R2, 3), "\n",
        "t50 = ", round(ann$t50, 1)
      )

      x_min <- min(df$time)
      x_max <- max(df$time)
      y_min <- min(df$Wo_minus_Wt)
      y_max <- max(df$Wo_minus_Wt)

      ann$x_pos <- c(x_min + 0.05*(x_max-x_min),
                     x_max - 0.05*(x_max-x_min))[seq_len(nrow(ann))]
      ann$y_pos <- c(y_max - 0.05*(y_max-y_min),
                     y_min + 0.05*(y_max-y_min))[seq_len(nrow(ann))]
      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(1, 0)[seq_len(nrow(ann))]

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
          hjust = ann$hjust,
          vjust = ann$vjust,
          size = 4,
          show.legend = FALSE
        )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
