% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/r2_nakagawa.R
\name{r2_nakagawa}
\alias{r2_nakagawa}
\title{Nakagawa's R2 for mixed models}
\usage{
r2_nakagawa(
  model,
  by_group = FALSE,
  tolerance = 1e-08,
  ci = NULL,
  iterations = 100,
  ci_method = NULL,
  null_model = NULL,
  approximation = "lognormal",
  model_component = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{A mixed effects model.}

\item{by_group}{Logical, if \code{TRUE}, returns the explained variance
at different levels (if there are multiple levels). This is essentially
similar to the variance reduction approach by \emph{Hox (2010), pp. 69-78}.}

\item{tolerance}{Tolerance for singularity check of random effects, to decide
whether to compute random effect variances for the conditional r-squared
or not. Indicates up to which value the convergence result is accepted. When
\code{r2_nakagawa()} returns a warning, stating that random effect variances
can't be computed (and thus, the conditional r-squared is \code{NA}),
decrease the tolerance-level. See also \code{\link[=check_singularity]{check_singularity()}}.}

\item{ci}{Confidence resp. credible interval level. For \code{icc()}, \code{r2()}, and
\code{rmse()}, confidence intervals are based on bootstrapped samples from the
ICC, R2 or RMSE value. See \code{iterations}.}

\item{iterations}{Number of bootstrap-replicates when computing confidence
intervals for the ICC, R2, RMSE etc.}

\item{ci_method}{Character string, indicating the bootstrap-method. Should
be \code{NULL} (default), in which case \code{lme4::bootMer()} is used for bootstrapped
confidence intervals. However, if bootstrapped intervals cannot be calculated
this way, try \code{ci_method = "boot"}, which falls back to \code{boot::boot()}. This
may successfully return bootstrapped confidence intervals, but bootstrapped
samples may not be appropriate for the multilevel structure of the model.
There is also an option \code{ci_method = "analytical"}, which tries to calculate
analytical confidence assuming a chi-squared distribution. However, these
intervals are rather inaccurate and often too narrow. It is recommended to
calculate bootstrapped confidence intervals for mixed models.}

\item{null_model}{Optional, a null model to compute the random effect variances,
which is passed to \code{\link[insight:get_variance]{insight::get_variance()}}. Usually only required if
calculation of r-squared or ICC fails when \code{null_model} is not specified. If
calculating the null model takes longer and you already have fit the null
model, you can pass it here, too, to speed up the process.}

\item{approximation}{Character string, indicating the approximation method
for the distribution-specific (observation level, or residual) variance. Only
applies to non-Gaussian models. Can be \code{"lognormal"} (default), \code{"delta"} or
\code{"trigamma"}. For binomial models, the default is the \emph{theoretical}
distribution specific variance, however, it can also be
\code{"observation_level"}. See \emph{Nakagawa et al. 2017}, in particular supplement
2, for details.}

\item{model_component}{For models that can have a zero-inflation component,
specify for which component variances should be returned. If \code{NULL} or
\code{"full"} (the default), both the conditional and the zero-inflation component
are taken into account. If \code{"conditional"}, only the conditional component is
considered.}

\item{verbose}{Toggle warnings and messages.}

\item{...}{Arguments passed down to \code{lme4::bootMer()} or \code{boot::boot()}
for bootstrapped ICC, R2, RMSE etc.; for \code{variance_decomposition()},
arguments are passed down to \code{brms::posterior_predict()}.}
}
\value{
A list with the conditional and marginal R2 values.
}
\description{
Compute the \emph{marginal} and \emph{conditional} r-squared value for
mixed effects models with complex random effects structures.
}
\details{
Marginal and conditional r-squared values for mixed models are calculated
based on \emph{Nakagawa et al. (2017)}. For more details on the computation of
the variances, see \code{\link[insight:get_variance]{insight::get_variance()}}. The random effect variances are
actually the mean random effect variances, thus the r-squared value is also
appropriate for mixed models with random slopes or nested random effects
(see \emph{Johnson, 2014}).
\itemize{
\item \strong{Conditional R2}: takes both the fixed and random effects into account.
\item \strong{Marginal R2}: considers only the variance of the fixed effects.
}

The contribution of random effects can be deduced by subtracting the
marginal R2 from the conditional R2 or by computing the \code{\link[=icc]{icc()}}.
}
\section{Supported models and model families}{

The single variance components that are required to calculate the marginal
and conditional r-squared values are calculated using the \code{\link[insight:get_variance]{insight::get_variance()}}
function. The results are validated against the solutions provided by
\emph{Nakagawa et al. (2017)}, in particular examples shown in the Supplement 2
of the paper. Other model families are validated against results from the
\strong{MuMIn} package. This means that the r-squared values returned by \code{r2_nakagawa()}
should be accurate and reliable for following mixed models or model families:
\itemize{
\item Bernoulli (logistic) regression
\item Binomial regression (with other than binary outcomes)
\item Poisson and Quasi-Poisson regression
\item Negative binomial regression (including nbinom1, nbinom2 and nbinom12 families)
\item Gaussian regression (linear models)
\item Gamma regression
\item Tweedie regression
\item Beta regression
\item Ordered beta regression
}

Following model families are not yet validated, but should work:
\itemize{
\item Zero-inflated and hurdle models
\item Beta-binomial regression
\item Compound Poisson regression
\item Generalized Poisson regression
\item Log-normal regression
\item Skew-normal regression
}

Extracting variance components for models with zero-inflation part is not
straightforward, because it is not definitely clear how the distribution-specific
variance should be calculated. Therefore, it is recommended to carefully
inspect the results, and probably validate against other models, e.g. Bayesian
models (although results may be only roughly comparable).

Log-normal regressions (e.g. \code{lognormal()} family in \strong{glmmTMB} or \code{gaussian("log")})
often have a very low fixed effects variance (if they were calculated as
suggested by \emph{Nakagawa et al. 2017}). This results in very low ICC or
r-squared values, which may not be meaningful.
}

\examples{
\dontshow{if (require("lme4")) withAutoprint(\{ # examplesIf}
model <- lme4::lmer(Sepal.Length ~ Petal.Length + (1 | Species), data = iris)
r2_nakagawa(model)
r2_nakagawa(model, by_group = TRUE)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Hox, J. J. (2010). Multilevel analysis: techniques and applications
(2nd ed). New York: Routledge.
\item Johnson, P. C. D. (2014). Extension of Nakagawa and Schielzeth’s R2 GLMM
to random slopes models. Methods in Ecology and Evolution, 5(9), 944–946.
\doi{10.1111/2041-210X.12225}
\item Nakagawa, S., and Schielzeth, H. (2013). A general and simple method for
obtaining R2 from generalized linear mixed-effects models. Methods in
Ecology and Evolution, 4(2), 133–142. \doi{10.1111/j.2041-210x.2012.00261.x}
\item Nakagawa, S., Johnson, P. C. D., and Schielzeth, H. (2017). The
coefficient of determination R2 and intra-class correlation coefficient from
generalized linear mixed-effects models revisited and expanded. Journal of
The Royal Society Interface, 14(134), 20170213.
}
}
