#' Create Book.txt file from files existing in quiz directory
#'
#' @param path path to the bookdown or quarto course repository, must have a `_bookdown.yml` or `_quarto.yml` file
#' @param md_files vector of file path of the md's to be included
#' @param output_dir output directory to put files.  It should likely be
#' relative to path
#' @param quiz_dir Where are the quizzes stored? Default looks for folder called "quizzes".
#' @param verbose print diagnostic messages
#'
#' @return A list of quiz and chapter files in order in a file called Book.txt -- How Leanpub wants it.
#' @export
#'
course_to_book_txt <- function(path = ".",
                               md_files = NULL,
                               output_dir = "manuscript",
                               quiz_dir = "quizzes",
                               verbose = TRUE) {
  # If md_files are not specified, then try to get them
  if (is.null(md_files)) {
    # Establish path
    root_dir <- course_path(path = path)

    rmd_regex <- "[.][q|R|r]md$"

    # Extract the names of the Rmd files (the chapters)
    md_files <- qrmd_files(path = root_dir)
  }

  if (!is.null(quiz_dir)) {
    # Find the quiz files in the quiz directory
    quiz_files <- list.files(pattern = "\\.md$", quiz_dir)

    # Put files in one vector
    all_files <- c(md_files, quiz_files)

    # Make a vector specifying the file type: quiz or not
    file_type <- c(
      rep("non-quiz", length(md_files)),
      rep("quiz", length(quiz_files))
    )
  } else {
    all_files <- md_files
    file_type <- rep("non-quiz", length(md_files))
  }
  # Put all files in one data.frame
  all_files <- data.frame(
    file_name = all_files,
    file_type
  ) %>%
    dplyr::mutate(
      # Use this so we don't have to fiddle with case senstivity for the next step
      lower_filename = tolower(file_name),
      # Get the number from the file name and that will be the order
      num = stringr::str_extract(file_name, "([0-9]+)"),
      num = dplyr::case_when(
        # Put index file first and about file last
        lower_filename == "index.rmd" ~ "0",
        lower_filename == "about.rmd" ~ as.character(length(all_files)),
        TRUE ~ num
      ),
      num = as.numeric(num),
      file_name = gsub(".Rmd$", ".md", file_name)
    ) %>%
    # Put quizzes in order!
    dplyr::arrange(num, file_type) %>%
    dplyr::pull(file_name)

  # Declare output file name
  book_txt <- file.path(output_dir, "Book.txt")

  if (verbose) {
    message(paste0("Autogenerated Book.txt saved to: ", book_txt))
  }
  # need to fix about quiz
  writeLines(all_files, con = book_txt)
}

#' Get file paths to all qmds or rmds in the course website directory
#'
#' @param path  Where to look for the _bookdown.yml or _quarto.yml file. Passes to get_yaml_spec() function. By default looks in current directory
#'
#' @return The file paths to rmds or wmds listed in the _bookdown.yml or _quarto.yml file.
#' @export
#'
qrmd_files <- function(path = ".") {
  yaml <- list.files(path = path, pattern = "_bookdown.yml|_quarto.yml", full.names = TRUE)

  spec <- yaml::read_yaml(yaml)

  rmd_files <- qmd_files <- NULL

  if (basename(yaml) == "_bookdown.yml") {
    rmd_files <- spec$rmd_files
  }
  if (basename(yaml) == "_quarto.yml") {
    qmd_files <- grep(".qmd", unlist(spec$book$chapters), value = TRUE)
  }
  if (length(rmd_files) > 0 && length(qmd_files) > 0) stop("Both qmd and rmd files are found. Not sure what format to expect")

  # Make files whichever ones exist here
  if (length(rmd_files) > 0) files <- rmd_files
  if (length(qmd_files) > 0) files <- qmd_files

  if (length(rmd_files) == 0 && length(qmd_files) == 0) {
    warning(
      "No rmd or qmd files found specified in the _quarto.yml or _bookdown.yml file. Going to try to find files in the repo based on suffix"
    )
    root_dir <- course_path(path = file.path(path))
    files <- list.files(
      pattern = "[.]Rmd$|[.]qmd$", ignore.case = TRUE,
      path = root_dir, full.names = FALSE
    )
  }

  # If we don't find files STOP
  if (length(files) == 0) stop("No rmd/qmd files found in the repo")

  return(files)
}
