% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nonprob.R, R/nonprob_documentation.R
\name{nonprob}
\alias{nonprob}
\title{Inference with Non-Probability Survey Samples}
\usage{
nonprob(
  data,
  selection = NULL,
  outcome = NULL,
  target = NULL,
  svydesign = NULL,
  pop_totals = NULL,
  pop_means = NULL,
  pop_size = NULL,
  method_selection = c("logit", "cloglog", "probit"),
  method_outcome = c("glm", "nn", "pmm", "npar"),
  family_outcome = c("gaussian", "binomial", "poisson"),
  subset = NULL,
  strata = NULL,
  case_weights = NULL,
  na_action = na.omit,
  control_selection = control_sel(),
  control_outcome = control_out(),
  control_inference = control_inf(),
  start_selection = NULL,
  start_outcome = NULL,
  verbose = FALSE,
  se = TRUE,
  ...
)
}
\arguments{
\item{data}{a \code{data.frame} with dataset containing the non-probability sample}

\item{selection}{a \code{formula} (default \code{NULL}) for the selection (propensity) score model}

\item{outcome}{a \code{formula} (default \code{NULL}) for the outcome (target) model}

\item{target}{a \code{formula} (default \code{NULL}) with target variable(s). We allow multiple target variables (e.g. \code{~y1 + y2 + y3})}

\item{svydesign}{an optional \code{svydesign2} class object containing a probability sample and design weights}

\item{pop_totals}{an optional \verb{named vector} with population totals of the covariates}

\item{pop_means}{an optional \verb{named vector} with population means of the covariates}

\item{pop_size}{an optional \code{double} value with population size}

\item{method_selection}{a \code{character} (default \code{logit}) indicating the method for the propensity score link function.}

\item{method_outcome}{a \code{character} (default \code{glm}) indicating the method for the outcome model.}

\item{family_outcome}{a \code{character} (default \code{gaussian})  describing the error distribution and the link function to be used in the model. Currently supports: \code{gaussian} with the identity link, \code{poisson} and \code{binomial}.}

\item{subset}{an optional \code{vector} specifying a subset of observations to be used in the fitting process}

\item{strata}{an optional \code{vector} specifying strata (not yet supported, for further development)}

\item{case_weights}{an optional \code{vector} of prior weights to be used in the fitting process.
It is assumed that this vector contains frequency or analytic weights (i.e. rows of the \code{data} argument are repeated according to the values of the \code{case_weights} argument), not probability/design weights.}

\item{na_action}{a function which indicates what should happen when the data contain \code{NAs} (default \code{na.omit} and it is the only method currently supported)}

\item{control_selection}{a \code{list} (default \code{control_sel()} result) indicating parameters to be used when fitting the selection model for propensity scores. To change the parameters one should use the \code{control_sel()} function}

\item{control_outcome}{a \code{list} (default \code{control_out()} result) indicating parameters to be used when fitting the model for the outcome variable. To change the parameters one should use the \code{control_out()} function}

\item{control_inference}{a \code{list} (default \code{control_inf()} result) indicating parameters to be used for inference based on probability and non-probability samples. To change the parameters one should use the \code{control_inf()} function}

\item{start_selection}{an optional \code{vector} with starting values for the parameters of the selection equation}

\item{start_outcome}{an optional \code{vector} with starting values for the parameters of the outcome equation}

\item{verbose}{a numerical value (default \code{TRUE}) whether detailed information on the fitting should be presented}

\item{se}{Logical value (default \code{TRUE}) indicating whether to calculate and return standard error of estimated mean}

\item{...}{Additional, optional arguments (not yet supported)}
}
\value{
Returns an object of the \code{nonprob} class (it is actually a \code{list}) which contains the following elements:\cr
\itemize{
\item{\code{call} -- the call of the \code{nonprob} function}
\item{\code{data} -- a \code{data.frame} passed from the \code{nonprob} function \code{data} argument}
\item{\code{X} -- a \code{model.matrix} containing data from probability (first \eqn{n_{S_B}} rows) and non-probability samples (next \eqn{n_{S_B}} rows) if specified at a function call}
\item{\code{y} -- a \code{list} of vector of outcome variables if specified at a function call}
\item{\code{R} -- a \verb{numeric vector} indicating whether a unit belongs to the probability (0) or non-probability (1) units in the matrix X}
\item{\code{ps_scores} -- a \verb{numeric vector} of estimated propensity scores for probability and non-probability sample}
\item{\code{case_weights} -- a \code{vector} of case weights for non-probability sample based on the call}
\item{\code{ipw_weights} -- a \code{vector} of inverse probability weights for non-probability sample (if applicable)}
\item{\code{control} -- a \code{list} of control functions based on the call}
\item{\code{output} -- a \code{data.frame} with the estimated means and standard errors for the variables specified in the \code{target} or \code{outcome} arguments}
\item{\code{SE} -- a \code{data.frame} with standard error of the estimator of the population mean, divided into errors from probability and non-probability samples (if applicable)}
\item{\code{confidence_interval} -- a \code{data.frame} with confidence interval of population mean estimator}
\item{\code{nonprob_size} -- a scalar \verb{numeric vector} denoting the size of non-probability sample}
\item{\code{prob_size} -- a scalar \verb{numeric vector} denoting the size of probability sample}
\item{\code{pop_size} -- a scalar \verb{numeric vector} estimated population size derived from estimated weights (non-probability sample) or known design weights (probability sample)}
\item{\code{pop_size_fixed} -- a \code{logical} value whether the population size was fixed (known) or estimated (unknown)}
\item{\code{pop_totals} -- a \verb{numeric vector} with the total values of the auxiliary variables derived from a probability sample or based on the call}
\item{\code{pop_means} -- a \verb{numeric vector} with the mean values of the auxiliary variables derived from a probability sample or based on the call}
\item{\code{outcome} -- a \code{list} containing information about the fitting of the mass imputation model. Structure of the object is based on the \code{method_outcome} and \code{family_outcome} arguments which point to specific methods as defined by functions \verb{method_*} (if specified in the call)}
\item{\code{selection} -- a \code{list} containing information about the fitting of the propensity score model. Structure of the object is based on the \code{method_selection} argument which point to specific methods as defined by functions \code{method_ps} (if specified in the call)}
\item{\code{boot_sample} -- a \code{matrix} with bootstrap estimates of the target variable(s) (if specified)}
\item{\code{svydesign} -- a \code{svydesign2} object (if specified)}
\item{\code{ys_rand_pred} -- a \code{list} of predicted values for the target variable(s) for the probability sample (for the MI and DR estimator)}
\item{\code{ys_nons_pred} -- a \code{list} of predicted values for the target variable(s) for the non-probability sample (for the MI and DR estimator)}
\item{\code{ys_resid} -- a \code{list} of residuals for the target variable(s) for the non-probability sample (for the MI and DR estimator)}
\item{\code{estimator} -- a \verb{character vector} with information what type of estimator was selected (one of \code{c("ipw", "mi", "dr")}).}
\item{\code{selection_formula} -- a \code{formula} based on the \code{selection} argument (if specified)}
\item{\code{estimator_method} -- a \verb{character vector} with information on the detailed method applied (for the \code{print} method)}
}
}
\description{
\code{nonprob} function provides an access to the various methods for inference based on non-probability surveys (including big data). The function allows to estimate the population mean based on the access to a reference probability sample (via the \code{survey} package),  as well as totals or means of covariates.

The package implements state-of-the-art approaches recently proposed in the literature: Chen et al. (2020),
Yang et al. (2020), Wu (2022) and uses the \href{https://CRAN.R-project.org/package=survey}{Lumley 2004} \code{survey} package for inference (if a reference probability sample is provided).

It provides various inverse probability weighting (e.g. with calibration constraints), mass imputation (e.g. nearest neighbour, predictive mean matching) and doubly robust estimators (e.g. that take into account minimisation of the asymptotic bias of the population mean estimators).

The package uses the \code{survey} package functionality when a probability sample is available.

All optional parameters are set to \code{NULL}. The obligatory ones include \code{data} as well as one of the following three:
\code{selection}, \code{outcome}, or \code{target} -- depending on which method has been selected.
In the case of \code{outcome} and \code{target} multiple \eqn{y} variables can be specified.
}
\details{
Let \eqn{y} be the response variable for which we want to estimate the population mean,
given by \deqn{\mu_{y} = \frac{1}{N} \sum_{i=1}^N y_{i}.} For this purpose we consider data integration
with the following structure. Let \eqn{S_A} be the non-probability sample with the design matrix of covariates as
\deqn{
\boldsymbol{X}_A =
  \begin{bmatrix}
x_{11} & x_{12} & \cdots & x_{1p} \cr
x_{21} & x_{22} & \cdots & x_{2p} \cr
\vdots & \vdots & \ddots & \vdots \cr
x_{n_{A1}} & x_{n_{A2}} & \cdots & x_{n_{Ap}} \cr
\end{bmatrix},
}
and vector of outcome variable
\deqn{
\boldsymbol{y} =
  \begin{bmatrix}
y_{1} \cr
y_{2} \cr
\vdots \cr
y_{n_{A}}
\end{bmatrix}.
}
On the other hand, let \eqn{S_B} be the probability sample with design matrix of covariates be
\deqn{
\boldsymbol{X}_B =
  \begin{bmatrix}
x_{11} & x_{12} & \cdots & x_{1p} \cr
x_{21} & x_{22} & \cdots & x_{2p} \cr
\vdots & \vdots & \ddots & \vdots \cr
x_{n_{B1}} & x_{n_{B2}} & \cdots & x_{n_{Bp}}\cr
\end{bmatrix}.
}
Instead of a sample of units we can consider a vector of population sums in the form of \eqn{\tau_x = (\sum_{i \in \mathcal{U}}\boldsymbol{x}_{i1}, \sum_{i \in \mathcal{U}}\boldsymbol{x}_{i2}, ..., \sum_{i \in \mathcal{U}}\boldsymbol{x}_{ip})} or means
\eqn{\frac{\tau_x}{N}}, where \eqn{\mathcal{U}} refers to a finite population. Note that we do not assume access to the response variable for \eqn{S_B}.
In general we make the following assumptions:
\enumerate{
\item The selection indicator of belonging to non-probability sample \eqn{R_{i}} and the response variable \eqn{y_i} are independent given the set of covariates \eqn{\boldsymbol{x}_i}.
\item All units have a non-zero propensity score, i.e., \eqn{\pi_{i}^{A} > 0} for all i.
\item The indicator variables \eqn{R_{i}^{A}} and \eqn{R_{j}^{A}} are independent for given \eqn{\boldsymbol{x}_i} and \eqn{\boldsymbol{x}_j} for \eqn{i \neq j}.
}

There are three possible approaches to the problem of estimating population mean using non-probability samples:
\enumerate{
\item Inverse probability weighting -- the main drawback of non-probability sampling is the unknown selection mechanism for a unit to be included in the sample.
This is why we talk about the so-called "biased sample" problem. The inverse probability approach is based on the assumption that a reference probability sample
is available and therefore we can estimate the propensity score of the selection mechanism.
The estimator has the following form:
\deqn{\hat{\mu}_{IPW} = \frac{1}{N^{A}}\sum_{i \in S_{A}} \frac{y_{i}}{\hat{\pi}_{i}^{A}}.}
For this purpose several estimation methods can be considered. The first approach is maximum likelihood estimation with a corrected
log-likelihood function, which is given by the following formula
\deqn{
 \ell^{*}(\boldsymbol{\theta}) = \sum_{i \in S_{A}}\log \left\lbrace \frac{\pi(\boldsymbol{x}_{i}, \boldsymbol{\theta})}{1 - \pi(\boldsymbol{x}_{i},\boldsymbol{\theta})}\right\rbrace + \sum_{i \in S_{B}}d_{i}^{B}\log \left\lbrace 1 - \pi({\boldsymbol{x}_{i},\boldsymbol{\theta})}\right\rbrace.}
In the literature, the main approach to modelling propensity scores is based on the logit link function.
However, we extend the propensity score model with the additional link functions such as cloglog and probit.
The pseudo-score equations derived from ML methods can be replaced by the idea of generalised estimating equations
with calibration constraints defined by equations.
\deqn{
 \mathbf{U}(\boldsymbol{\theta})=\sum_{i \in S_A} \mathbf{h}\left(\mathbf{x}_i, \boldsymbol{\theta}\right)-\sum_{i \in S_B} d_i^B \pi\left(\mathbf{x}_i, \boldsymbol{\theta}\right) \mathbf{h}\left(\mathbf{x}_i, \boldsymbol{\theta}\right).}
Notice that for \eqn{ \mathbf{h}\left(\mathbf{x}_i, \boldsymbol{\theta}\right) = \frac{\pi(\boldsymbol{x}, \boldsymbol{\theta})}{\boldsymbol{x}}} We do not need a probability sample and can use a vector of population totals/means.
\item Mass imputation -- This method is based on a framework where imputed values of outcome variables are created for the entire probability sample. In this case, we treat the large sample as a training data set that is used to build an imputation model.
Using the imputed values for the probability sample and the (known) design weights,
we can build a population mean estimator of the form:
\deqn{\hat{\mu}_{MI} = \frac{1}{N^B}\sum_{i \in S_{B}} d_{i}^{B} \hat{y}_i.}
It opens the door to a very flexible method for imputation models. The package uses generalized linear models from \code{\link[stats:glm]{stats::glm()}},
the nearest neighbour algorithm using \code{\link[RANN:nn2]{RANN::nn2()}} and predictive mean matching.
\item Doubly robust estimation -- The IPW and MI estimators are sensitive to misspecified models for the propensity score and outcome variable, respectively.
To this end, so-called doubly robust methods are presented that take these problems into account.
It is a simple idea to combine propensity score and imputation models during inference, leading to the following estimator
\deqn{\hat{\mu}_{DR} = \frac{1}{N^A}\sum_{i \in S_A} \hat{d}_i^A (y_i - \hat{y}_i) + \frac{1}{N^B}\sum_{i \in S_B} d_i^B \hat{y}_i.}
In addition, an approach based directly on bias minimisation has been implemented. The following formula
\deqn{
   \begin{aligned}
   bias(\hat{\mu}_{DR}) = & \mathbb{E} (\hat{\mu}_{DR} - \mu) \cr = & \mathbb{E} \left\lbrace \frac{1}{N} \sum_{i=1}^N (\frac{R_i^A}{\pi_i^A (\boldsymbol{x}_i^{\mathrm{T}} \boldsymbol{\theta})}
   - 1 ) (y_i - \operatorname{m}(\boldsymbol{x}_i^{\mathrm{T}} \boldsymbol{\beta})) \right\rbrace \cr + & \mathbb{E} \left\lbrace \frac{1}{N} \sum_{i=1}^N (R_i^B d_i^B - 1) \operatorname{m}( \boldsymbol{x}_i^{\mathrm{T}} \boldsymbol{\beta}) \right\rbrace,
   \end{aligned}
   }
lead us to system of equations
\deqn{
   \begin{aligned}
   J(\theta, \beta) =
   \left\lbrace
   \begin{array}{c}
                      J_1(\theta, \beta) \cr
                      J_2(\theta, \beta)
                      \end{array}\right\rbrace = \left\lbrace \begin{array}{c}
                                               \sum_{i=1}^N R_i^A\ \left\lbrace \frac{1}{\pi(\boldsymbol{x}_i, \boldsymbol{\theta})}-1 \right\rbrace \left\lbrace y_i-m(\boldsymbol{x}_i, \boldsymbol{\beta}) \right\rbrace \boldsymbol{x}_i \cr
                                               \sum_{i=1}^N \frac{R_i^A}{\pi(\boldsymbol{x}_i, \boldsymbol{\theta})} \frac{\partial m(\boldsymbol{x}_i, \boldsymbol{\beta})}{\partial \boldsymbol{\beta}}
                                               - \sum_{i \in \mathcal{S}_{\mathrm{B}}} d_i^{\mathrm{B}} \frac{\partial m(\boldsymbol{x}_i, \boldsymbol{\beta})}{\partial \boldsymbol{\beta}}
  \end{array} \right\rbrace,
  \end{aligned}
  }
where \eqn{m\left(\boldsymbol{x}_{i}, \boldsymbol{\beta}\right)} is a mass imputation (regression) model for the outcome variable and
propensity scores \eqn{\pi_i^A} are estimated using a \code{logit} function for the model. As with the \code{MLE} and \code{GEE} approaches we have extended
this method to \code{cloglog} and \code{probit} links.
}

As it is not straightforward to calculate the variances of these estimators, asymptotic equivalents of the variances derived using the Taylor approximation have been proposed in the literature.
Details can be found \href{https://ncn-foreigners.ue.poznan.pl/nonprobsvy-book/}{here}.
In addition, the bootstrap approach can be used for variance estimation.

The function also allows variables selection using known methods that have been implemented to handle the integration of probability and non-probability sampling.
In the presence of high-dimensional data, variable selection is important, because it can reduce the variability in the estimate that results from using irrelevant variables to build the model.
Let \eqn{\operatorname{U}\left( \boldsymbol{\theta}, \boldsymbol{\beta} \right)} be the joint estimating function for \eqn{\left( \boldsymbol{\theta}, \boldsymbol{\beta} \right)}. We define the
penalized estimating functions as
\deqn{
  \operatorname{U}^p \left(\boldsymbol{\theta}, \boldsymbol{\beta}\right) =
  \operatorname{U}\left(\boldsymbol{\theta}, \boldsymbol{\beta}\right) -
  \left\lbrace
  \begin{array}{c}
  q_{\lambda_\theta}(|\boldsymbol{\theta}|) \operatorname{sgn}(\boldsymbol{\theta}) \\
  q_{\lambda_\beta}(|\boldsymbol{\beta}|) \operatorname{sgn}(\boldsymbol{\beta})
  \end{array}
  \right\rbrace,
  }
where \eqn{\lambda_{\theta}} and \eqn{q_{\lambda_{\beta}}} are some smooth functions. We let \eqn{q_{\lambda} \left(x\right) = \frac{\partial p_{\lambda}}{\partial x}}, where \eqn{p_{\lambda}} is some penalization function.
Details of penalization functions and techniques for solving this type of equation can be found \href{https://ncn-foreigners.ue.poznan.pl/nonprobsvy-book/}{here}.
To use the variable selection model, set the \code{vars_selection} parameter in the \code{\link[=control_inf]{control_inf()}} function to \code{TRUE}. In addition, in the other control functions such as \code{\link[=control_sel]{control_sel()}} and \code{\link[=control_out]{control_out()}}
you can set parameters for the selection of the relevant variables, such as the number of folds during cross-validation algorithm or the lambda value for penalizations. Details can be found
in the documentation of the control functions for \code{nonprob}.
}
\examples{
\donttest{
# generate data based on Doubly Robust Inference With Non-probability Survey Samples (2021)
# Yilin Chen , Pengfei Li & Changbao Wu
set.seed(123)
# sizes of population and probability sample
N <- 20000 # population
n_b <- 1000 # probability
# data
z1 <- rbinom(N, 1, 0.7)
z2 <- runif(N, 0, 2)
z3 <- rexp(N, 1)
z4 <- rchisq(N, 4)

# covariates
x1 <- z1
x2 <- z2 + 0.3 * z2
x3 <- z3 + 0.2 * (z1 + z2)
x4 <- z4 + 0.1 * (z1 + z2 + z3)
epsilon <- rnorm(N)
sigma_30 <- 10.4
sigma_50 <- 5.2
sigma_80 <- 2.4

# response variables
y30 <- 2 + x1 + x2 + x3 + x4 + sigma_30 * epsilon
y50 <- 2 + x1 + x2 + x3 + x4 + sigma_50 * epsilon
y80 <- 2 + x1 + x2 + x3 + x4 + sigma_80 * epsilon

# population
sim_data <- data.frame(y30, y50, y80, x1, x2, x3, x4)
## propensity score model for non-probability sample (sum to 1000)
eta <- -4.461 + 0.1 * x1 + 0.2 * x2 + 0.1 * x3 + 0.2 * x4
rho <- plogis(eta)

# inclusion probabilities for probability sample
z_prob <- x3 + 0.2051
sim_data$p_prob <- n_b* z_prob/sum(z_prob)

# data
sim_data$flag_nonprob <- as.numeric(runif(N) < rho) ## sampling nonprob
sim_data$flag_prob <- as.numeric(runif(n_b) < sim_data$p_prob) ## sampling prob
nonprob_df <- subset(sim_data, flag_nonprob == 1) ## non-probability sample
svyprob <- svydesign(
  ids = ~1, probs = ~p_prob,
  data = subset(sim_data, flag_prob == 1),
  pps = "brewer"
) ## probability sample

## mass imputation estimator
mi_res <- nonprob(
  outcome = y30 + y50 + y80 ~ x1 + x2 + x3 + x4,
  data = nonprob_df,
  svydesign = svyprob
)
mi_res
## inverse probability weighted estimator
ipw_res <- nonprob(
  selection = ~ x1 + x2 + x3 + x4,
  target = ~y30 + y50 + y80,
  data = nonprob_df,
  svydesign = svyprob
)
ipw_res
## doubly robust estimator
dr_res <- nonprob(
  outcome = y30 + y50 + y80 ~ x1 + x2 + x3 + x4,
  selection = ~ x1 + x2 + x3 + x4,
  data = nonprob_df,
  svydesign = svyprob
)
dr_res
}
}
\references{
Kim JK, Park S, Chen Y, Wu C. Combining non-probability and
probability survey samples through mass imputation. J R Stat Soc Series A. 2021;184:941–
963.

Shu Yang, Jae Kwang Kim, Rui Song. Doubly robust inference when combining probability
and non-probability samples with high dimensional data. J. R. Statist. Soc. B (2020)

Yilin Chen , Pengfei Li & Changbao Wu (2020) Doubly Robust Inference
With Nonprobability Survey Samples, Journal of the American Statistical Association, 115:532,
2011-2021

Shu Yang, Jae Kwang Kim and Youngdeok Hwang Integration of data from
probability surveys and big found data for finite population inference using mass imputation.
Survey Methodology, June 2021 29 Vol. 47, No. 1, pp. 29-58
}
\seealso{
\code{\link[stats:optim]{stats::optim()}} -- For more information on the \code{optim} function used in the
\code{optim} method of propensity score model fitting.

\code{\link[maxLik:maxLik]{maxLik::maxLik()}} -- For more information on the \code{maxLik} function used in
\code{maxLik} method of propensity score model fitting.

\code{\link[ncvreg:cv.ncvreg]{ncvreg::cv.ncvreg()}} -- For more information on the \code{cv.ncvreg} function used in
variable selection for the outcome model.

\code{\link[nleqslv:nleqslv]{nleqslv::nleqslv()}} -- For more information on the \code{nleqslv} function used in
estimation process of the bias minimization approach.

\code{\link[stats:glm]{stats::glm()}} -- For more information about the generalised linear models used during mass imputation process.

\code{\link[RANN:nn2]{RANN::nn2()}} -- For more information about the nearest neighbour algorithm used during mass imputation process.

\code{\link[=control_sel]{control_sel()}} -- For the control parameters related to selection model.

\code{\link[=control_out]{control_out()}} -- For the control parameters related to outcome model.

\code{\link[=control_inf]{control_inf()}} -- For the control parameters related to statistical inference.
}
\author{
Łukasz Chrostowski, Maciej Beręsewicz, Piotr Chlebicki
}
