% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tcc.R
\name{tcc}
\alias{tcc}
\alias{print.tcc}
\title{Apply a treatment-choice criterion (TCC) to get treatment preferences 
based on network meta-analysis estimates.}
\usage{
tcc(
  x,
  pooled = if (x$random) "random" else "common",
  swd = NULL,
  swd.below.null = NULL,
  swd.above.null = NULL,
  small.values = x$small.values,
  relax = TRUE,
  level = x$level.ma
)

\method{print}{tcc}(x, ...)
}
\arguments{
\item{x}{A \code{\link[netmeta]{netmeta}} object.}

\item{pooled}{A character string indicating whether results for the
common (\code{"common"}) or random effects model
(\code{"random"}) should be used. Can be abbreviated. If not specified the
results from the random effects model will be used by default.}

\item{swd}{A numeric value specifying the smallest worthwhile difference
value (SWD); see Details.}

\item{swd.below.null}{A numeric value specifying the SWD below the null
effect (see Details).}

\item{swd.above.null}{A numeric value specifying the SWD above the null
effect (see Details).}

\item{small.values}{A character string specifying whether small 
treatment effects indicate a beneficial (\code{"desirable"}) or
harmful (\code{"undesirable"}) effect.}

\item{relax}{A logical optional argument. If TRUE (default), the treatment
choice criterion is based solely on the SWD bounds, emphasizing only the
clinical importance of the results. If set to FALSE, the criterion
incorporates both statistical significance and clinical importance.
We recommend using the default setting (see Details).}

\item{level}{The level used to calculate confidence intervals for
log-abilities.}

\item{\dots}{Additional arguments (ignored).}
}
\value{
NMA estimates in a preference format.
}
\description{
This function uses a treatment choice criterion defined by the user and
transforms the network meta-analysis estimates into a preference format that
indicates either a treatment preference or a tie. In this setting, a
treatment preference implies that the respective NMA estimate represents
a clinically important result (i.e. that fulfills the TCC) while a tie
indicates that the respective NMA estimate lacks enough evidence to represent
a treatment preference. The resulting preference format is then used as input 
to \code{\link{mtrank}}.
}
\details{
R function \code{\link{mtrank}} expects data in a \bold{preference}
format, where a treatment preference or tie is indicated for each network
meta-analysis (NMA) estimate. For example, for the comparison between
treatments \emph{A} and \emph{B} the potential outcomes are:
\itemize{
\item \emph{A} > \emph{B}
\item \emph{A} < \emph{B}
\item \emph{A} = \emph{B}
}

The transformation takes place based on the NMA estimates and the treatment
choice criterion which has the form of a decision rule.

This function implements treatment choice criteria based on the range of
equivalence (ROE) which are specified by 
\itemize{
\item argument \code{swd}. Then the limits of the ROE
  will be defined based on the values (i) \code{swd}, \code{1 / swd} for
  ratio measures and (ii) \code{swd} and \code{-swd} for difference
  measures.
\item arguments \code{swd.below.null} and \code{swd.above.null}.
  These arguments allow the users to define their own limits of the ROE,
  given the restriction that the lower limit will always be smaller than the
  upper limit.
}

Note that when the argument \code{swd} is specified, the arguments
\code{swd.below.null} and \code{swd.above.null} are ignored.
Either only the \code{swd} or both of the \code{swd.below.null} and
\code{swd.above.null} must be specified for the proper
definition of the ROE.

After setting the ROE, each NMA treatment effect will be categorised as a
treatment preference or a tie. The argument \code{relax} controls the amount
of conservatism of the treatment choice criterion. If set to \code{FALSE},
a TCC will be built requiring both clinical importance as statistical
significance of the results. If set to \code{TRUE} (default), the criterion
uses only the ROE bounds and therefore the NMA treatment effects need to be
only clinically important to indicate a treatment preference.
}
\examples{
data("antidepressants")
#
pw1 <- pairwise(studlab = studyid, treat = drug_name,
  n = ntotal, event = responders,
  data = antidepressants, sm = "OR")
# Use subset to reduce runtime
pw0 <- subset(pw1, studyid < 60)
#
net0 <- netmeta(pw0, reference.group = "tra")

ranks0 <- tcc(net0, swd = 1.20, small.values = "undesirable")

# Comparison other drugs vs trazodone
forest(ranks0,
  label.left = "Favours trazodone",
  label.right = "Favours other drug")

# Comparison escitalopram vs other drugs
forest(ranks0, reference.group = "esc", baseline = FALSE,
  label.left = "Favours other drug",
  label.right = "Favours escitalopram")

\dontrun{
# Store a PDF file in the current working directory showing all results
# (this is the default, i.e., if argument 'reference.group' is missing)
forest(ranks0, baseline = FALSE, reference.group = trts,
  file = "forest_tcc_antidepressants.pdf")

# Run analysis with full data set
net1 <- netmeta(pw1, reference.group = "tra")

ranks1 <- tcc(net1, swd = 1.20, small.values = "undesirable")

# Comparison other drugs vs trazodone
forest(ranks1,
  label.left = "Favours trazodone",
  label.right = "Favours other drug")

# Comparison escitalopram vs other drugs
forest(ranks1, reference.group = "esc", baseline = FALSE,
  label.left = "Favours other drug",
  label.right = "Favours escitalopram")
}

}
\references{
Evrenoglou T, Nikolakopoulou A, Schwarzer G, Rücker G, Chaimani A (2024):
Producing treatment hierarchies in network meta-analysis using probabilistic
models and treatment-choice criteria,
\url{https://arxiv.org/abs/2406.10612}
}
