% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apply_imputation.R
\name{apply_imputation}
\alias{apply_imputation}
\title{Apply a function for imputation}
\usage{
apply_imputation(
  ds,
  FUN = mean,
  type = "columnwise",
  convert_tibble = TRUE,
  ...
)
}
\arguments{
\item{ds}{A data frame or matrix with missing values.}

\item{FUN}{The function to be applied for imputation.}

\item{type}{A string specifying the values used for imputation; one of:
"columnwise", "rowwise", "total", "Two-Way" or "Winer"  (see details).}

\item{convert_tibble}{If \code{ds} is a tibble, should it be converted
(see section A note for tibble users).}

\item{...}{Further arguments passed to \code{FUN}.}
}
\value{
An object of the same class as \code{ds} with imputed missing values.
}
\description{
Apply a function for imputation over rows, columns or combinations of both
}
\details{
The functionality of \code{apply_imputation} is inspired by the
\code{\link[base]{apply}} function. The function applies a function
\code{FUN} to impute the missing values in \code{ds}. \code{FUN} must be a
function, which takes a vector as input and returns exactly one value. The
argument \code{type} is comparable to \code{\link[base]{apply}}'s
\code{MARGIN} argument. It specifies the values that are used for the
calculation of the imputation values. For example, \code{type = "columnwise"}
and \code{FUN = mean} will impute the mean of the observed values in a column
for all missing values in this column. In contrast, \code{type = "rowwise"}
and \code{FUN = mean} will impute the mean of the observed values in a row
for all missing values in this row.

List of all implemented \code{types}:
\itemize{
\item{"columnwise" (the default): imputes column by column; all observed
values of a column are given  to \code{FUN} and the returned value is used
as the imputation value for all missing values of the column.}
\item{"rowwise": imputes row by row; all observed values of a row are given
to \code{FUN} and the returned value is used as the imputation value for all
missing values of the row.}
\item{"total": All observed values of \code{ds} are given to \code{FUN} and
the returned value is used as the imputation value for all missing values of
\code{ds}.}
\item{"Winer": The mean value from "columnwise" and "rowwise" is used as the
imputation value.}
\item{"Two-Way": The sum of the values from "columnwise" and "rowwise" minus
"total" is used as the imputation value.}
}

If no value can be given to \code{FUN} (for example, if no value in a column
is observed and \code{type = "columnwise"}), then a warning will be issued
and no value will be imputed in the corresponding column or row.
}
\section{A note for tibble users}{

If you use tibbles and \code{convert_tibble} is \code{TRUE} the tibble is
first converted to a data frame, then imputed and converted back. If
\code{convert_tibble} is \code{FALSE} no conversion is done. However,
depending on the tibble and the package version of tibble you use,
imputation may not be possible and some errors will be thrown.
}

\examples{
ds <- data.frame(X = 1:20, Y = 101:120)
ds_mis <- delete_MCAR(ds, 0.2)
ds_imp_app <- apply_imputation(ds_mis, FUN = mean, type = "total")
# the same result can be achieved via impute_mean():
ds_imp_mean <- impute_mean(ds_mis, type = "total")
all.equal(ds_imp_app, ds_imp_mean)
}
\references{
Beland, S., Pichette, F., & Jolani, S. (2016). Impact on
  Cronbach's \eqn{\alpha}{alpha} of simple treatment methods for missing
  data. \emph{The Quantitative Methods for Psychology}, 12(1), 57-73.
}
\seealso{
A convenient interface exists for common cases like mean imputation:
  \code{\link{impute_mean}}, \code{\link{impute_median}},
  \code{\link{impute_mode}}. All these functions
  call \code{apply_imputation}.
}
