% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msl.R
\name{msl}
\alias{msl}
\alias{msw}
\alias{msf}
\title{Compute MCTQ local time of mid-sleep}
\usage{
msl(so, sd)
}
\arguments{
\item{so}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time of
sleep onset} from a standard, micro, or shift version of the MCTQ
questionnaire. You can use \code{\link[=so]{so()}} to compute it for the
standard or shift version.}

\item{sd}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration} from a standard, micro, or shift version of the MCTQ
questionnaire. You can use \code{\link[=sdu]{sdu()}} to compute it for any
MCTQ version.}
}
\value{
An \code{\link[hms:hms]{hms}} object corresponding to the vectorized sum of
\code{so} and \code{(sd / 2)} in a circular time frame of 24 hours.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{msl()} computes the \strong{local time of mid-sleep} for standard, micro, and
shift versions of the Munich ChronoType Questionnaire (MCTQ).

Please note that, although we tried to preserve the original authors' naming
pattern for the MCTQ functions, the name \code{ms} provokes a dangerous name
collision with the \code{\link[lubridate:hms]{ms()}} function (a function for parsing
minutes and seconds components). That's why we named it \code{msl}. \code{msl()} and
\code{\link[=sdu]{sdu()}} are the only exceptions, all the other \code{mctq}
functions maintain a strong naming resemblance with the original authors'
naming pattern.
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


Roenneberg, Allebrandt, Merrow, & Vetter (2012), Ghotbi et al. (2020), Juda,
Vetter, & Roenneberg (2013), and The Worldwide Experimental Platform (n.d.)
guidelines for \code{msl()} (\eqn{MSW} or \eqn{MSF}) computation are as follows.
\subsection{Notes}{
\itemize{
\item This computation must be applied to each section of the questionnaire.
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{For standard and micro versions of the MCTQ}{

\deqn{MS_{W/F} = SO_{W/F} + \frac{SD_{W/F}}{2}}

Where:
\itemize{
\item \eqn{MS_{W/F}} = Local time of mid-sleep on work \strong{or} work-free days.
\item \eqn{SO_{W/F}} = Local time of sleep onset on work \strong{or} work-free
days.
\item \eqn{SD_{W/F}} = Sleep duration on work \strong{or} work-free days.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days.
}

\subsection{For the shift version of the MCTQ}{

\deqn{MS_{W/F}^{M/E/N} = SO_{W/F}^{M/E/N} + \frac{SD_{W/F}^{M/E/N}}{2}}

Where:
\itemize{
\item \eqn{MS_{W/F}^{M/E/N}}{SO_W/F_M/E/N} = Local time of mid-sleep between
two days in a particular shift \strong{or} between two free days after a
particular shift.
\item \eqn{SO_{W/F}^{M/E/N}}{SO_W/F_M/E/N} = Local time of sleep onset between
two days in a particular shift \strong{or} between two free days after a
particular shift.
\item \eqn{SD_{W/F}^{M/E/N}}{SD_W/F_M/E/N} = Sleep duration between two days in a
particular shift \strong{or} between two free days after a particular shift.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days, \eqn{M} =
Morning shift; \eqn{E} = Evening shift; \eqn{N} = Night shift.
}
}

\examples{
## Scalar example

so <- hms::parse_hm("23:30")
sd <- lubridate::dhours(8)
msl(so, sd)
#> 03:30:00 # Expected

so <- hms::parse_hm("01:00")
sd <- lubridate::dhours(10)
msl(so, sd)
#> 06:00:00 # Expected

so <- hms::as_hms(NA)
sd <- lubridate::dhours(7.5)
msl(so, sd)
#> NA # Expected

## Vector example

so <- c(hms::parse_hm("00:10"), hms::parse_hm("01:15"))
sd <- c(lubridate::dhours(9.25), lubridate::dhours(5.45))
msl(so, sd)
#> [1] 04:47:30 # Expected
#> [1] 03:58:30 # Expected
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
\seealso{
Other MCTQ functions: 
\code{\link{fd}()},
\code{\link{gu}()},
\code{\link{le_week}()},
\code{\link{msf_sc}()},
\code{\link{napd}()},
\code{\link{sd24}()},
\code{\link{sd_overall}()},
\code{\link{sd_week}()},
\code{\link{sdu}()},
\code{\link{sjl_sc}()},
\code{\link{sjl_weighted}()},
\code{\link{sjl}()},
\code{\link{so}()},
\code{\link{tbt}()}
}
\concept{MCTQ functions}
