% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bland_altman_repeated.R
\name{bland_altman_repeated}
\alias{bland_altman_repeated}
\alias{print.ba_repeated}
\alias{print.ba_repeated_matrix}
\alias{plot.ba_repeated}
\alias{plot.ba_repeated_matrix}
\title{Bland-Altman for repeated measurements}
\usage{
bland_altman_repeated(
  data = NULL,
  response,
  subject,
  method,
  time,
  two = 1.96,
  conf_level = 0.95,
  include_slope = FALSE,
  use_ar1 = FALSE,
  ar1_rho = NA_real_,
  max_iter = 200L,
  tol = 1e-06,
  verbose = FALSE
)

\method{print}{ba_repeated}(x, digits = 3, ci_digits = 3, ...)

\method{print}{ba_repeated_matrix}(x, digits = 3, ci_digits = 3, style = c("pairs", "matrices"), ...)

\method{plot}{ba_repeated}(
  x,
  title = "Bland-Altman (repeated measurements)",
  subtitle = NULL,
  point_alpha = 0.7,
  point_size = 2.2,
  line_size = 0.8,
  shade_ci = TRUE,
  shade_alpha = 0.08,
  smoother = c("none", "loess", "lm"),
  symmetrize_y = TRUE,
  show_points = TRUE,
  ...
)

\method{plot}{ba_repeated_matrix}(
  x,
  pairs = NULL,
  against = NULL,
  facet_scales = c("free_y", "fixed"),
  title = "Bland-Altman (repeated, pairwise)",
  point_alpha = 0.6,
  point_size = 1.8,
  line_size = 0.7,
  shade_ci = TRUE,
  shade_alpha = 0.08,
  smoother = c("none", "loess", "lm"),
  show_points = TRUE,
  ...
)
}
\arguments{
\item{data}{Optional \code{data.frame}/\code{data.table} containing required columns.}

\item{response}{Numeric vector (stacked outcomes) \strong{or} a single character
string giving the column name in \code{data}.}

\item{subject}{Subject ID (integer/factor/numeric) \strong{or} a single character
string giving the column name in \code{data}.}

\item{method}{Method label (factor/character/integer; N >= 2 levels) \strong{or}
a single character string giving the column name in \code{data}.}

\item{time}{Integer/numeric replicate/time index (pairs within subject) \strong{or}
a single character string giving the column name in \code{data}.}

\item{two}{Positive scalar; LoA multiple of SD (default 1.96).}

\item{conf_level}{Confidence level for CIs (default 0.95).}

\item{include_slope}{Logical. If \code{TRUE}, the model includes the pair mean
as a fixed effect and estimates a proportional-bias slope.
Look at details for deeper information. We recommend fit both with and without the slope. If \eqn{\beta_1} is
materially non-zero over a wide level range, consider a scale transformation
(e.g., log or percent-logit) and re-fit without the slope.}

\item{use_ar1}{Logical; AR(1) within-subject residual correlation.}

\item{ar1_rho}{AR(1) parameter (|rho|<1).}

\item{max_iter, tol}{EM control for the backend (defaults 200, 1e-6).}

\item{verbose}{Logical; print brief progress.}

\item{x}{A \code{"ba_repeated_matrix"} object.}

\item{digits}{Number of digits for estimates (default 3).}

\item{ci_digits}{Number of digits for CI bounds (default 3).}

\item{...}{Additional theme adjustments passed to \code{ggplot2::theme(...)}
(e.g., \code{plot.title.position = "plot"}, \code{axis.title.x = element_text(size=11)}).}

\item{style}{Show as pairs or matrix format?}

\item{title}{Plot title (character scalar). Defaults to
\code{"Bland-Altman (repeated measurements)"} for two methods and
\code{"Bland-Altman (repeated, pairwise)"} for the faceted matrix plot.}

\item{subtitle}{Optional subtitle (character scalar). If \code{NULL}, a compact
summary is shown using the fitted object.}

\item{point_alpha}{Numeric in \verb{[0, 1]}. Transparency for scatter points
drawn at (pair mean, pair difference) when point data are available.
Passed to \code{ggplot2::geom_point(alpha = ...)}. Default \code{0.7}.}

\item{point_size}{Positive numeric. Size of scatter points; passed to
\code{ggplot2::geom_point(size = ...)}. Default \code{2.2}.}

\item{line_size}{Positive numeric. Line width for horizontal bands
(bias and both LoA) and, when requested, the proportional-bias line.
Passed to \code{ggplot2::geom_hline(linewidth = ...)} (and \code{geom_abline}).
Default \code{0.8}.}

\item{shade_ci}{Logical. If \code{TRUE} and confidence intervals are available in
the object (\code{CI.lines} for two methods; \verb{*_ci_*} matrices for the pairwise
case), semi-transparent rectangles are drawn to indicate CI bands for the
bias and each LoA. If \code{FALSE}, dashed horizontal CI lines are drawn instead.
Has no effect if CIs are not present. Default \code{TRUE}.}

\item{shade_alpha}{Numeric in \verb{[0, 1]}. Opacity of the CI shading
rectangles when \code{shade_ci = TRUE}. Passed to \code{ggplot2::annotate("rect", alpha = ...)}.
Default \code{0.08}.}

\item{smoother}{One of \code{"none"}, \code{"loess"}, or \code{"lm"}. Adds an overlaid trend
for differences vs means when points are drawn, to visualise proportional
bias. \code{"lm"} fits a straight line with no SE ribbon; \code{"loess"} draws a
locally-smoothed curve (span 0.9) with no SE ribbon; \code{"none"} draws no
smoother. Ignored if \code{show_points = FALSE} or if no point data are available.}

\item{symmetrize_y}{Logical (two-method plot only). If \code{TRUE}, the y-axis is
centred at the estimated bias and expanded symmetrically to cover all
elements used to compute the range (bands, CIs, and points if shown).
Default \code{TRUE}.}

\item{show_points}{Logical. If \code{TRUE}, per-pair points are drawn when present
in the fitted object (two-method path) or when they can be reconstructed
from \code{x$data_long} and \code{x$mapping} (pairwise path). If \code{FALSE} or if point
data are unavailable, only the bands (and optional CI indicators) are drawn.
Default \code{TRUE}.}

\item{pairs}{(Faceted pairwise plot only.) Optional character vector of
labels specifying which method contrasts to display. Labels must match the
"row - column" convention used by \code{print()}/\code{summary()} (e.g., \code{"B - A"}).
Defaults to all upper-triangle pairs.}

\item{against}{(Faceted pairwise plot only.) Optional single method name.
If supplied, facets are restricted to contrasts of the chosen method
against all others. Ignored when \code{pairs} is provided.}

\item{facet_scales}{(Faceted pairwise plot only.) Either \code{"free_y"} (default)
to allow each facet its own y-axis limits, or \code{"fixed"} for a common scale
across facets. Passed to \code{ggplot2::facet_wrap(scales = ...)}.}
}
\value{
Either a \code{"ba_repeated"} object (exactly two methods) or a
\code{"ba_repeated_matrix"} object (pairwise results when \eqn{\ge}3 methods).

\strong{If \code{"ba_repeated_matrix"} (N\eqn{\ge}3 methods)}, outputs are:
\itemize{
\item \code{bias} \eqn{(m \times m)}; estimated mean difference (row - column).
Diagonal is \code{NA}.
\item \code{sd_loa} \eqn{(m \times m)}; estimated SD of a \emph{single new} paired
difference for the (row, column) methods, accounting for the random
subject intercept and (if enabled) AR(1) correlation.
\item \code{loa_lower}, \code{loa_upper} \eqn{(m \times m)}; limits of agreement
for a single measurement pair, computed as
\eqn{\mathrm{bias} \pm \mathrm{two}\times \mathrm{sd\_loa}}.
Signs follow the row - column convention
(e.g., \code{loa_lower[j,i] = -loa_upper[i,j]}).
\item \code{width} \eqn{(m \times m)}; LoA width,
\code{loa_upper - loa_lower} (= \code{2 * two * sd_loa}).
\item \code{n} \eqn{(m \times m)}; number of subject-time pairs used in each
pairwise BA (complete cases where both methods are present).

\item \strong{CI matrices at nominal \code{conf_level}} (delta method):
\itemize{
\item \code{mean_ci_low}, \code{mean_ci_high}; CI for the bias.
\item \code{loa_lower_ci_low}, \code{loa_lower_ci_high}; CI for the lower LoA.
\item \code{loa_upper_ci_low}, \code{loa_upper_ci_high}; CI for the upper LoA.
}

\item \code{slope} (\eqn{m \times m}; optional); proportional-bias slope (difference
vs pair mean) when \code{include_slope = TRUE}.

\item \code{sigma2_subject} \eqn{(m \times m)}; estimated variance of the
subject-level random intercept (on differences).
\item \code{sigma2_resid} \eqn{(m \times m)}; estimated residual variance of a
single difference after accounting for the random intercept (and AR(1), if used).

\item \code{use_ar1} \emph{(scalar logical)}; whether AR(1) modeling was requested.
\item \code{ar1_rho} \emph{(scalar numeric or \code{NA})}; user-supplied \eqn{\rho}
if a single common value was provided; \code{NA} otherwise.
\item \code{ar1_rho_pair} (\eqn{m \times m}; optional); \eqn{\rho} actually used per pair
(may be estimated from data or equal to the supplied value). AR(1) blocks
require consecutive integer time indices for each subject.
\item \code{ar1_estimated} (\eqn{m \times m}; optional logical); for each pair, \code{TRUE}
if \eqn{\rho} was estimated internally; \code{FALSE} if supplied.

\item \code{methods} \emph{(character)}; method level names; matrix rows/columns
follow this order.
\item \code{two} \emph{(scalar)}; LoA multiplier used (default \code{1.96}).
\item \code{conf_level} \emph{(scalar)}; nominal confidence level used for CIs.

\item \code{data_long} \emph{(data.frame)}; the long data used for fitting
(\code{response}, \code{subject}, \code{method}, \code{time}). Included to
facilitate plotting/reproducibility; not required for summary methods.
}

\strong{If \code{"ba_repeated"} (exactly two methods)}, outputs are:
\itemize{
\item \code{mean.diffs} \emph{(scalar)}; estimated bias (method 2 - method 1).
\item \code{lower.limit}, \code{upper.limit} \emph{(scalars)}; LoA
\eqn{\mu \pm \mathrm{two}\times \mathrm{SD}} for a single new pair.
\item \code{critical.diff} \emph{(scalar)}; \code{two * SD}; LoA half-width.
\item \code{two}, \code{conf_level} \emph{(scalars)}; as above.
\item \code{CI.lines} \emph{(named numeric)}; CI bounds for bias and both LoA
(\code{*.ci.lower}, \code{*.ci.upper}) at \code{conf_level}.
\item \code{means}, \code{diffs} \emph{(vectors)}; per-pair means and differences
used by plotting helpers.
\item \code{based.on} \emph{(integer)}; number of subject-time pairs used.
\item \code{include_slope}, \code{beta_slope}; whether a proportional-bias slope
was estimated and its value (if requested).
\item \code{sigma2_subject}, \code{sigma2_resid}; variance components as above.
\item \code{use_ar1}, \code{ar1_rho}, \code{ar1_estimated}; AR(1) settings/results
as above (scalars for the two-method fit).
}
}
\description{
Repeated-measures Bland-Altman (BA) for method comparison based on a
mixed-effects model fitted to \strong{subject-time matched paired differences}.
A subject-specific random intercept accounts for clustering, and (optionally)
an AR(1) process captures serial correlation across replicates within subject.
The function returns bias (mean difference), limits of agreement (LoA),
confidence intervals, and variance components, for either two methods or
all pairwise contrasts when \eqn{\ge}3 methods are supplied.

\strong{Required columns / vectors}
\itemize{
\item \code{response}: numeric measurements.
\item \code{subject}: subject identifier (integer/factor/numeric).
\item \code{method}: method label with \eqn{\ge}2 levels (factor/character/integer).
\item \code{time}: replicate index used to form pairs; only records where
both methods are present for the same \code{subject} and \code{time}
contribute to a given pairwise BA (rows with missing components are dropped
before fitting each pair).
}
}
\details{
The function implements a repeated-measures Bland–Altman (BA) analysis for
two or more methods using a linear mixed model fitted to
\emph{subject–time matched paired differences}. For any selected pair of
methods \eqn{(a,b)}, let
\deqn{d_{it} = y_{itb} - y_{ita}, \qquad m_{it} = \tfrac{1}{2}(y_{itb} + y_{ita}),}
where \eqn{y_{itm}} denotes the observed value from method \eqn{m\in\{a,b\}} on
subject \eqn{i} at replicate/time \eqn{t}; \eqn{d_{it}} is the paired difference
(method \eqn{b} minus method \eqn{a}); and \eqn{m_{it}} is the corresponding pair mean.
Here \eqn{i=1,\ldots,S} indexes subjects and \eqn{t} indexes replicates/time within subject.
Only records with both methods present at the same \code{subject} and \code{time}
contribute to that pair.

The fitted per-pair model is
\deqn{d_{it} = \beta_0 + \beta_1\, m_{it} + u_i + \varepsilon_{it},}
with random intercept \eqn{u_i \sim \mathcal{N}(0, \sigma_u^2)} and
within-subject residual vector \eqn{\varepsilon_i} having covariance
\eqn{\operatorname{Cov}(\varepsilon_i) = \sigma_e^2\, \mathbf{R}_i}.
When \code{use_ar1 = FALSE}, \eqn{\mathbf{R}_i = \mathbf{I}_{n_i}} (i.i.d.).
When \code{use_ar1 = TRUE}, \eqn{\mathbf{R}_i = \mathbf{C}_i^{-1}} and
\eqn{\mathbf{C}_i} encodes an AR(1) \emph{precision} structure over time
(see below). Setting \code{include_slope = FALSE} drops the regressor
\eqn{m_{it}} (i.e., \eqn{\beta_1 = 0}).

\subsection{AR(1) within-subject correlation}{
For each subject, observations are first ordered by \code{time}. Contiguous
blocks satisfy \eqn{t_{k+1} = t_k + 1}; non-contiguous gaps and any
negative/NA times are treated as singletons (i.i.d.).

For a contiguous block of length \eqn{L} and AR(1) parameter \eqn{\rho}
(clipped to \eqn{(-0.999, 0.999)}), the blockwise \emph{precision} matrix
\eqn{\mathbf{C}} has entries
\deqn{\mathbf{C} = \frac{1}{1-\rho^2}\,\begin{bmatrix}
1 & -\rho &       &        &  \\
-\rho & 1+\rho^2 & -\rho  &        &  \\
      & \ddots   & \ddots & \ddots &  \\
      &          & -\rho  & 1+\rho^2 & -\rho \\
      &          &        & -\rho    & 1
\end{bmatrix}.}
The full \eqn{\mathbf{C}_i} is block-diagonal over contiguous segments, with
a small ridge added to the diagonal for numerical stability. Residual
covariance is \eqn{\sigma_e^2 \mathbf{R}_i = \sigma_e^2 \mathbf{C}_i^{-1}}.

If \code{use_ar1 = TRUE} and \code{ar1_rho} is \code{NA}, \eqn{\rho} is
estimated in two passes (i) fit the i.i.d. model; (ii) compute detrended
residuals within each contiguous block (remove block-specific intercept and
linear time), form lag-1 correlations, apply a small-sample bias adjustment
\eqn{(1-\rho^2)/L}, pool with Fisher's \eqn{z} (weights \eqn{\approx L-3}),
and refit with the pooled \eqn{\hat\rho}.
}

\subsection{Estimation by stabilised EM/GLS}{
Let \eqn{\mathbf{X}_i} be the fixed-effects design for subject \eqn{i}
(intercept, and optionally the pair mean). Internally, the pair mean
regressor is centred and scaled before fitting to stabilise the slope;
estimates are back-transformed to the original units afterwards.

Given current \eqn{(\sigma_u^2, \sigma_e^2)}, the marginal precision of
\eqn{d_i} integrates \eqn{u_i} via a Woodbury/Sherman-Morrison identity:
\deqn{\mathbf{V}_i^{-1} \;=\; \sigma_e^{-2}\mathbf{C}_i \;-\;
\sigma_e^{-2}\mathbf{C}_i \mathbf{1}\,
\Big(\sigma_u^{-2} + \sigma_e^{-2}\mathbf{1}^\top \mathbf{C}_i \mathbf{1}\Big)^{-1}
\mathbf{1}^\top \mathbf{C}_i \sigma_e^{-2}.}
The GLS update is
\deqn{\hat\beta \;=\; \Big(\sum_i \mathbf{X}_i^\top \mathbf{V}_i^{-1} \mathbf{X}_i\Big)^{-1}
\Big(\sum_i \mathbf{X}_i^\top \mathbf{V}_i^{-1} d_i\Big).}
Writing \eqn{r_i = d_i - \mathbf{X}_i\hat\beta}, the E-step gives the BLUP of
the random intercept and its conditional second moment:
\deqn{M_i \;=\; \sigma_u^{-2} + \sigma_e^{-2}\mathbf{1}^\top \mathbf{C}_i \mathbf{1},\qquad
\tilde u_i \;=\; M_i^{-1}\,\sigma_e^{-2}\mathbf{1}^\top \mathbf{C}_i r_i,\qquad
\mathbb{E}[u_i^2\mid y] \;=\; \tilde u_i^2 + M_i^{-1}.}
The M-step updates the variance components by
\deqn{\hat\sigma_u^2 \;=\; \frac{1}{m}\sum_i \mathbb{E}[u_i^2\mid y], \qquad
\hat\sigma_e^2 \;=\; \frac{1}{n}\sum_i \Big\{(r_i - \mathbf{1}\tilde u_i)^\top
\mathbf{C}_i (r_i - \mathbf{1}\tilde u_i) + M_i^{-1}\,\mathbf{1}^\top \mathbf{C}_i \mathbf{1}\Big\}.}
The algorithm employs positive-definite inverses with ridge fallback,
trust-region damping of variance updates (ratio-bounded), and clipping of
parameters to prevent degeneracy.
}

\subsection{Point estimates reported}{
The reported BA \emph{bias} is the \emph{subject-equal} mean of the paired
differences,
\deqn{\mu_0 \;=\; \frac{1}{m}\sum_{i=1}^m \bar d_i,\qquad
\bar d_i = \frac{1}{n_i}\sum_{t=1}^{n_i} d_{it},}
which is robust to unbalanced replicate counts (\eqn{n_i}) and coincides with
the GLS intercept when subjects contribute equally. If \code{include_slope =
TRUE}, the proportional-bias slope is estimated against the pair mean
\eqn{m_{it}}; internally \eqn{m_{it}} is centred and scaled and the slope is
returned on the original scale.
}

\subsection{Proportional bias slope (\code{include_slope})}{
When \code{include_slope = TRUE}, the model augments the mean difference with
the pair mean \eqn{m_{it} = \tfrac{1}{2}(y_{itb}+y_{ita})}:
\deqn{d_{it} = \beta_0 + \beta_1 m_{it} + u_i + \varepsilon_{it}.}
Internally \eqn{m_{it}} is centred and scaled for numerical stability; the
reported \code{beta_slope} and intercept are back-transformed to the original scale.

\eqn{\beta_1 \neq 0} indicates level-dependent (proportional)
bias, where the difference between methods changes with the overall measurement level.
The Bland–Altman limits of agreement produced by this function remain the
conventional, \emph{horizontal} bands centred at the subject-equal bias \eqn{\mu_0};
they are not regression-adjusted LoA.

On an appropriate scale and when
the two methods have comparable within-subject variances, the null expectation is
\eqn{\beta_1 \approx 0}. However, because \eqn{m_{it}} contains measurement error
from both methods, unequal precisions can produce a spurious slope even if there is
no true proportional bias. Under a simple no-bias data-generating model
\eqn{y_{itm}=\theta_{it}+c_m+e_{itm}} with independent errors of variances
\eqn{\sigma_a^2,\sigma_b^2}, the expected OLS slope is approximately
\deqn{\mathbb{E}[\hat\beta_1] \;\approx\;
\dfrac{\tfrac{1}{2}(\sigma_b^2-\sigma_a^2)}
      {\mathrm{Var}(\theta_{it}) + \tfrac{1}{4}(\sigma_a^2+\sigma_b^2)},}
showing zero expectation when \eqn{\sigma_a^2=\sigma_b^2} and attenuation as the
level variability \eqn{\mathrm{Var}(\theta_{it})} increases.
}

\subsection{Limits of Agreement (LoA)}{
A \emph{single new paired measurement} for a random subject has variance
\deqn{\mathrm{Var}(d^\star) \;=\; \sigma_u^2 + \sigma_e^2,}
so the LoA are
\deqn{\mathrm{LoA} \;=\; \mu_0 \;\pm\; \texttt{two}\,\sqrt{\sigma_u^2 + \sigma_e^2}.}
The argument \code{two} is the SD multiplier (default \eqn{z_{1-\alpha/2}}
implied by \code{conf_level}, e.g., \eqn{1.96} at \eqn{95\%}).
}

\subsection{Wald/Delta confidence intervals}{
CIs for \emph{bias} and each LoA use a delta method around the EM estimates.
The standard error of \eqn{\mu_0} is based on the dispersion of subject
means \eqn{\mathrm{Var}(\mu_0) \approx \mathrm{Var}(\bar d_i)/m}. For
\eqn{\mathrm{sd}_{\mathrm{LoA}} = \sqrt{V}} with \eqn{V=\sigma_u^2+\sigma_e^2},
we can define
\deqn{\mathrm{Var}(\mathrm{sd}) \;\approx\; \frac{\mathrm{Var}(V)}{4V}, \quad
\mathrm{Var}(V) \;\approx\; \mathrm{Var}(\hat\sigma_u^2) + \mathrm{Var}(\hat\sigma_e^2)
+ 2\,\mathrm{Cov}(\hat\sigma_u^2,\hat\sigma_e^2).}
The per-subject contributions used to approximate these terms are:
\eqn{A_i=\mathbb{E}[u_i^2\mid y]} and
\eqn{B_i=\{(r_i-\mathbf{1}\tilde u_i)^\top\mathbf{C}_i(r_i-\mathbf{1}\tilde u_i)
+ M_i^{-1}\mathbf{1}^\top\mathbf{C}_i\mathbf{1}\}/n_i}. Empirical variances
of \eqn{A_i} and \eqn{B_i} (with replicate-size weighting for \eqn{B_i}) and
their covariance yield \eqn{\mathrm{Var}(\hat\sigma_u^2)},
\eqn{\mathrm{Var}(\hat\sigma_e^2)} and \eqn{\mathrm{Cov}(\hat\sigma_u^2,\hat\sigma_e^2)}.
For a LoA bound \eqn{L_\pm = \mu_0 \pm \texttt{two}\cdot \mathrm{sd}}, the
working variance is
\deqn{\mathrm{Var}(L_\pm) \;\approx\; \mathrm{Var}(\mu_0) + \texttt{two}^2\,\mathrm{Var}(\mathrm{sd}),}
and Wald CIs use the normal quantile at \code{conf_level}. These are
large-sample approximations; with very small \eqn{m} they may be conservative.
}

\subsection{Three or more methods}{
When \eqn{\ge 3} methods are supplied, the routine performs the above fit for
each unordered pair \eqn{(j,k)} and reassembles matrices. Specifically,
\itemize{
\item \emph{Orientation} is defined to be \emph{row minus column}, i.e.,
\eqn{\texttt{bias}[j,k]} estimates \eqn{\mathbb{E}(y_k - y_j)}.
\item \eqn{\texttt{bias}} is antisymmetric
(\eqn{b_{jk} = -b_{kj}}); \eqn{\texttt{sd\_loa}} and \eqn{\texttt{width}}
are symmetric; LoA obey \eqn{\texttt{loa\_lower}[j,k] = -\texttt{loa\_upper}[k,j]}.
\item \eqn{\texttt{n}[j,k]} is the number of subject-time pairs
used for that contrast (complete cases where both methods are present).
}
}

\subsection{Missingness, time irregularity, and safeguards}{
Pairs are formed only where both methods are observed at the same
\code{subject} and \code{time}; other records do not influence that pair.
AR(1) structure is applied only over contiguous time sequences within
subject; gaps break contiguity and revert those positions to i.i.d.
Numerically, inverses use a positive-definite solver with adaptive ridge and
pseudo-inverse fallback; variance updates are clamped and ratio-damped;
\eqn{\rho} is clipped to \eqn{(-0.999,0.999)}.
}
}
\examples{
# -------- Simulate repeated-measures data --------
set.seed(1)

# design (no AR)
# subjects
S   <- 30L
# replicates per subject
Tm  <- 15L
subj <- rep(seq_len(S), each = Tm)
time <- rep(seq_len(Tm), times = S)

# subject signal centered at 0 so BA "bias" won't be driven by the mean level
mu_s  <- rnorm(S, mean = 0, sd = 8)
# constant within subject across replicates
true  <- mu_s[subj]

# common noise (no AR, i.i.d.)
sd_e <- 2
e0   <- rnorm(length(true), 0, sd_e)

# --- Methods ---
# M1: signal + noise
y1 <- true + e0

# M2: same precision as M1; here identical so M3 can be
#     almost perfectly the inverse of both M1 and M2
y2 <- y1 + rnorm(length(true), 0, 0.01)

# M3: perfect inverse of M1 and M2
y3 <- -y1   # = -(true + e0)

# M4: unrelated to all others (pure noise, different scale)
y4 <- rnorm(length(true), 3, 6)

data <- rbind(
  data.frame(y = y1, subject = subj, method = "M1", time = time),
  data.frame(y = y2, subject = subj, method = "M2", time = time),
  data.frame(y = y3, subject = subj, method = "M3", time = time),
  data.frame(y = y4, subject = subj, method = "M4", time = time)
)
data$method <- factor(data$method, levels = c("M1","M2","M3","M4"))

# quick sanity checks
with(data, {
  Y <- split(y, method)
  round(cor(cbind(M1 = Y$M1, M2 = Y$M2, M3 = Y$M3, M4 = Y$M4)), 3)
})

# Run BA (no AR)
ba4 <- bland_altman_repeated(
  data = data,
  response = "y", subject = "subject", method = "method", time = "time",
  two = 1.96, conf_level = 0.95,
  include_slope = FALSE, use_ar1 = FALSE
)
summary(ba4)
plot(ba4)

# -------- Simulate repeated-measures with AR(1) data --------
set.seed(123)
S <- 40L                      # subjects
Tm <- 50L                     # replicates per subject
methods <- c("A","B","C")     # N = 3 methods
rho <- 0.4                    # AR(1) within-subject across time

ar1_sim <- function(n, rho, sd = 1) {
  z <- rnorm(n)
  e <- numeric(n)
  e[1] <- z[1] * sd
  if (n > 1) for (t in 2:n) e[t] <- rho * e[t-1] + sqrt(1 - rho^2) * z[t] * sd
  e
}

# Subject baseline + time trend (latent "true" signal)
subj <- rep(seq_len(S), each = Tm)
time <- rep(seq_len(Tm), times = S)
# subject effects
mu_s  <- rnorm(S, 50, 7)
trend <- rep(seq_len(Tm) - mean(seq_len(Tm)), times = S) * 0.8
true  <- mu_s[subj] + trend

# Method-specific biases (B has +1.5 constant; C has slight proportional bias)
bias  <- c(A = 0, B = 1.5, C = -0.5)
# proportional component on "true"
prop  <- c(A = 0.00, B = 0.00, C = 0.10)

# Build long data: for each method, add AR(1) noise within subject over time
make_method <- function(meth, sd = 3) {
  e <- unlist(lapply(split(seq_along(time), subj),
                     function(ix) ar1_sim(length(ix), rho, sd)))
  y <- true * (1 + prop[meth]) + bias[meth] + e
  data.frame(y = y, subject = subj, method = meth, time = time,
             check.names = FALSE)
}

data <- do.call(rbind, lapply(methods, make_method))
data$method <- factor(data$method, levels = methods)

# -------- Repeated BA (pairwise matrix) ---------------------
baN <- bland_altman_repeated(
  response = data$y, subject = data$subject, method = data$method, time = data$time,
  two = 1.96, conf_level = 0.95,
  include_slope = FALSE,         # estimate proportional bias per pair
  use_ar1 = TRUE # model AR(1) within-subject
)

# Matrices (row - column orientation)
print(baN)
summary(baN)

# Faceted BA scatter by pair
plot(baN, smoother = "lm", facet_scales = "free_y")

# -------- Two-method path (A vs B only) -----------------------------------
data_AB <- subset(data, method \%in\% c("A","B"))
baAB <- bland_altman_repeated(
  response = data_AB$y, subject = data_AB$subject,
  method = droplevels(data_AB$method), time = data_AB$time,
  include_slope = FALSE, use_ar1 = TRUE, ar1_rho = 0.4
)
print(baAB)
plot(baAB)

}
\author{
Thiago de Paula Oliveira
}
