# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' @md
#' @name stable_pdf_and_cdf
#' @aliases stable_cdf stable_pdf
#' @title PDF and CDF of a skew stable distribution.
#' @description Evaluate the PDF or the CDF of the skew stable distribution with parameters
#' pars = c(alpha, beta, sigma, mu) at the points given in x.\cr\cr
#' _parametrization_ argument specifies the parametrization used for the distribution
#' as described by JP Nolan (1997). The default value is _parametrization_ = 0.\cr\cr
#' _tol_ sets the relative error tolerance (precision) to _tol_. The default value is tol = 1e-12.
#' @param x Vector of points where the pdf will be evaluated.
#' @param pars Vector with an initial estimation of the parameters. `pars_init = c(alpha, beta, sigma, mu)`, where
#' * alpha: shape / stability parameter, with 0 < alpha <= 2.
#' * beta: skewness parameter, with -1 <= beta <= 1.
#' * sigma: scale parameter, with 0 < sigma.
#' * mu: location parameter, with mu real.
#' @param parametrization Parametrization used for the skew stable distribution, as defined by JP Nolan (1997). By default, parametrization = 0.
#' @param tol Relative error tolerance (precission) of the calculated values. By default, tol = 1e-12.
#' @return A numeric vector.
#' @author Javier Royuela del Val, Federico Simmross Wattenberg and Carlos Alberola López\cr\cr
#'         Maintainer: Javier Royuela del Val <jroyval@@lpi.tel.uva.es>
#' @references Nolan JP (1997). Numerical Calculation of Stable Densities and Distribution Functions. Stochastic Models, 13(4) 759-774.
#' @keywords distribution
#' @examples
#' pars <- c(1.5, 0.9, 1, 0)
#' x <- seq(-5, 10, 0.001)
#'
#' pdf <- stable_pdf(x, pars)
#' cdf <- stable_cdf(x, pars)
#'
#' plot(x, pdf, type = "l")
#' @export
stable_pdf <- function(x, pars, parametrization = 0L, tol = 1e-12) {
    .Call(`_libstable4u_stable_pdf`, x, pars, parametrization, tol)
}

#' @export
stable_cdf <- function(x, pars, parametrization = 0L, tol = 1e-12) {
    .Call(`_libstable4u_stable_cdf`, x, pars, parametrization, tol)
}

#' @md
#' @title Quantile function of skew stable distributions
#' @description Evaluate the quantile function (CDF^-1) of the skew stable distribution
#' with parameters pars = c(alpha, beta, sigma, mu) at the points given in p.\cr\cr
#' _parametrization_ argument specifies the parametrization used for the distribution
#' as described by JP Nolan (1997). The default value is _parametrization_ = 0.\cr\cr
#' _tol_ sets the relative error tolerance (precission) to _tol_. The default value is tol = 1e-12.
#' @param p Vector of points where the quantile function will be evaluated, with  0 < p\[i] < 1.0
#' @param pars Vector with an initial estimation of the parameters. `pars_init = c(alpha, beta, sigma, mu)`, where
#' * alpha: shape / stability parameter, with 0 < alpha <= 2.
#' * beta: skewness parameter, with -1 <= beta <= 1.
#' * sigma: scale parameter, with 0 < sigma.
#' * mu: location parameter, with mu real.
#' @param parametrization Parametrization used for the skew stable distribution, as defined by JP Nolan (1997). By default, parametrization = 0.
#' @param tol Relative error tolerance (precission) of the calculated values. By default, tol = 1e-12.
#' @return A numeric vector.
#' @author Javier Royuela del Val, Federico Simmross Wattenberg and Carlos Alberola López\cr\cr
#'         Maintainer: Javier Royuela del Val <jroyval@@lpi.tel.uva.es>
#' @keywords distribution
#' @export
stable_q <- function(p, pars, parametrization = 0L, tol = 1e-12) {
    .Call(`_libstable4u_stable_q`, p, pars, parametrization, tol)
}

#' @md
#' @title Skew stable distribution random sample generation.
#' @description `stable_rnd(N, pars)` generates N random samples of a skew stable distribuiton
#' with parameters pars = c(alpha, beta, sigma, mu) using the Chambers, Mallows,
#' and Stuck (1976) method.\cr\cr
#' @param N Number of values to generate.
#' @param pars Vector with an initial estimation of the parameters. `pars_init = c(alpha, beta, sigma, mu)`, where
#' * alpha: shape / stability parameter, with 0 < alpha <= 2.
#' * beta: skewness parameter, with -1 <= beta <= 1.
#' * sigma: scale parameter, with 0 < sigma.
#' * mu: location parameter, with mu real.
#' @param parametrization Parametrization used for the skew stable distribution, as defined by JP Nolan (1997). By default, parametrization = 0.
#' @references Chambers JM, Mallows CL, Stuck BW (1976). A Method for Simulating Stable Random Variables. Journal of the American Statistical Association, 71(354), 340-344. doi:10.1080/01621459.1976.10480344.
#' @return A numeric vector.
#' @author Javier Royuela del Val, Federico Simmross Wattenberg and Carlos Alberola López\cr\cr
#'         Maintainer: Javier Royuela del Val <jroyval@@lpi.tel.uva.es>
#' @keywords distribution
#' @examples
#' N <- 1000
#' pars <- c(1.25, 0.95, 1.0, 0.0)
#' rnd <- stable_rnd(N, pars)
#'
#' hist(rnd)
#' @export
stable_rnd <- function(N, pars, parametrization = 0L) {
    .Call(`_libstable4u_stable_rnd`, N, pars, parametrization)
}

#' @md
#' @title Methods for parameter estimation of skew stable distributions.
#' @name stable_fit
#' @aliases stable_fit_init stable_fit_koutrouvelis stable_fit_mle stable_fit_mle2d
#' @description A set of functions are provided that perform the parameter estimation of skew stable distributions with different methods.
#' @details
#' * `stable_fit_init()` uses McCulloch's method of quantiles \[3]. This is usually a good initialization for the rest of the methods.
#' * `stable_fit_koutrouvelis()` implements Koutrouvellis' method based on the characteristic function \[4].
#' * `stable_fit_mle()` implements a Maximum likelihood estimation.
#' * `stable_fit_mle2()` implements a modified maximum likelihood estimation as described in \[1].
#' @param rnd Random sample
#' @param parametrization Parametrization used for the skew stable distribution, as defined by JP Nolan (1997). By default, parametrization = 0.
#' @return A numeric vector.
#' @references
#' * \[1] Royuela-del-Val J, Simmross-Wattenberg F, Alberola López C (2017). libstable: Fast, Parallel and High-Precision Computation of alpha-stable Distributions in R, C/C++ and MATLAB. Journal of Statistical Software, 78(1), 1-25. doi:10.18637/jss.v078.i01
#' * \[2] Chambers JM, Mallows CL, Stuck BW (1976). A Method for Simulating Stable Random Variables. Journal of the American Statistical Association, 71(354), 340-344. doi:10.1080/01621459.1976.10480344.
#' * \[3] McCulloch JH (1986). Simple Consistent Estimators of Stable Distribution Parameters. Communications in Statistics - Simulation and Computation, 15(4), 1109-1136. doi:10.1080/03610918608812563.
#' * \[4] Koutrouvelis IA (1981). An Iterative Procedure for the Estimation of the Parameters of Stable Laws. Communications in Statistics - Simulation and Computation, 10(1), 17-28. doi:10.1080/03610918108812189.
#' * \[5] Nolan JP (1997). Numerical Calculation of Stable Densities and Distribution Functions. Stochastic Models, 13(4) 759-774. doi:10.1080/15326349708807450.
#' @keywords distribution
#' @author Javier Royuela del Val, Federico Simmross Wattenberg and Carlos Alberola López\cr\cr
#'         Maintainer: Javier Royuela del Val <jroyval@@lpi.tel.uva.es>
#' @examples
#' # Set alpha, beta, sigma and mu stable parameters in a vector
#' pars <- c(1.5, 0.9, 1, 0)
#'
#' # Generate random values
#' set.seed(1)
#' rnd <- stable_rnd(100, pars)
#' head(rnd)
#'
#' # Estimate the parameters of the skew stable distribution given
#' # the generated sample:
#'
#' # Using the McCulloch's estimator:
#' pars_init <- stable_fit_init(rnd)
#'
#' # Using the Koutrouvelis' estimator, with McCulloch estimation
#' # as a starting point:
#' pars_est_K <- stable_fit_koutrouvelis(rnd, pars_init)
#'
#' # Using maximum likelihood estimator:
#' pars_est_ML <- stable_fit_mle(rnd, pars_est_K)
#'
#' # Using modified maximum likelihood estimator (see [1]):
#' pars_est_ML2 <- stable_fit_mle2d(rnd, pars_est_K)
#' @export
stable_fit_init <- function(rnd, parametrization = 0L) {
    .Call(`_libstable4u_stable_fit_init`, rnd, parametrization)
}

#' @md
#' @rdname stable_fit
#' @param pars_init Vector with an initial estimation of the parameters. `pars_init = c(alpha, beta, sigma, mu)`, where
#' * alpha: shape / stability parameter, with 0 < alpha <= 2.
#' * beta: skewness parameter, with -1 <= beta <= 1.
#' * sigma: scale parameter, with 0 < sigma.
#' * mu: location parameter, with mu real.
#' @export
stable_fit_koutrouvelis <- function(rnd, pars_init = as.numeric( c()), parametrization = 0L) {
    .Call(`_libstable4u_stable_fit_koutrouvelis`, rnd, pars_init, parametrization)
}

#' @export
stable_fit_mle <- function(rnd, pars_init = as.numeric( c()), parametrization = 0L) {
    .Call(`_libstable4u_stable_fit_mle`, rnd, pars_init, parametrization)
}

#' @export
stable_fit_mle2d <- function(rnd, pars_init = as.numeric( c()), parametrization = 0L) {
    .Call(`_libstable4u_stable_fit_mle2d`, rnd, pars_init, parametrization)
}

