% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kre_adaptive.R
\name{bias_ES2012}
\alias{bias_ES2012}
\title{Bias Estimator of Eichner & Stute (2012)}
\usage{
bias_ES2012(sigma, h, xXh, thetaXh, K, mmDiff)
}
\arguments{
\item{sigma}{Numeric vector \eqn{(\sigma_1, \ldots, \sigma_s)} with
\eqn{s \ge 1} with values of the scale parameter \eqn{\sigma}.}

\item{h}{Numeric scalar for bandwidth \eqn{h} (as ``contained'' in
\code{thetaXh} and \code{xXh}).}

\item{xXh}{Numeric vector expecting the pre-computed h-scaled differences
\eqn{(x - X_1)/h}, \ldots, \eqn{(x - X_n)/h} where \eqn{x} is the
single (!) location for which the weights are to be computed,
the \eqn{X_i}'s are the data values, and \eqn{h} is the numeric
bandwidth scalar.}

\item{thetaXh}{Numeric vector expecting the pre-computed h-scaled differences
\eqn{(\theta - X_1)/h}, \ldots, \eqn{(\theta - X_n)/h} where
\eqn{\theta} is the numeric scalar location parameter, and the
\eqn{X_i}'s and \eqn{h} are as in \code{xXh}.}

\item{K}{A kernel function (with vectorized in- & output) to be used for the
estimator.}

\item{mmDiff}{Numeric vector expecting the pre-computed differences
\eqn{m_n(X_1) - m_n(x), \ldots, m_n(X_n) - m_n(x)}.}
}
\value{
A numeric vector of the length of \code{sigma}.
}
\description{
Bias estimator \eqn{Bias_n(\sigma)}, vectorized in \eqn{\sigma}, on p. 2540
of Eichner & Stute (2012).
}
\details{
The formula can also be found in eq. (15.21) of Eichner (2017).
Pre-computed \eqn{(x - X_i)/h}, \eqn{(\theta - X_i)/h}, and
\eqn{m_n(X_i) - m_n(x)} are expected for efficiency reasons (and are
currently prepared in function \code{\link{kare}}).
}
\examples{
require(stats)

 # Regression function:
m <- function(x, x1 = 0, x2 = 8, a = 0.01, b = 0) {
 a * (x - x1) * (x - x2)^3 + b
}
 # Note: For a few details on m() see examples in ?nadwat.

n <- 100       # Sample size.
set.seed(42)   # To guarantee reproducibility.
X <- runif(n, min = -3, max = 15)      # X_1, ..., X_n   # Design.
Y <- m(X) + rnorm(length(X), sd = 5)   # Y_1, ..., Y_n   # Response.

h <- n^(-1/5)
Sigma <- seq(0.01, 10, length = 51)   # sigma-grid for minimization.
x0 <- 5   # Location at which the estimator of m should be computed.

 # m_n(x_0) and m_n(X_i) for i = 1, ..., n:
mn <- nadwat(x = c(x0, X), dataX = X, dataY = Y, K = dnorm, h = h)

 # Estimator of Bias_x0(sigma) on the sigma-grid:
(Bn <- bias_ES2012(sigma = Sigma, h = h, xXh = (x0 - X) / h,
  thetaXh = (mean(X) - X) / h, K = dnorm, mmDiff = mn[-1] - mn[1]))

\dontrun{
 # Visualizing the estimator of Bias_n(sigma) at x on the sigma-grid:
plot(Sigma, Bn, type = "o", xlab = expression(sigma), ylab = "",
  main = bquote(widehat("Bias")[n](sigma)~~"at"~~x==.(x0)))
}

}
\references{
Eichner & Stute (2012) and Eichner (2017): see \code{\link{kader}}.
}
\seealso{
\code{\link{kare}} which currently does the pre-computing.
}
