% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit_volume}
\alias{fit_volume}
\title{Fit a Univariate State-Space Model on Intraday Trading Volume}
\usage{
fit_volume(
  data,
  fixed_pars = NULL,
  init_pars = NULL,
  verbose = 0,
  control = NULL
)
}
\arguments{
\item{data}{An n_bin * n_day matrix or an \code{xts} object storing intraday trading volume.}

\item{fixed_pars}{A list of parameters' fixed values. The allowed parameters are listed below,
\itemize{\item{\code{"a_eta"}: \eqn{a^{\eta}}{a.\eta}} of size 1 ;
         \item{\code{"a_mu"}: \eqn{a^{\mu}}{a.\mu}} of size 1 ;
         \item{\code{"var_eta"}: \eqn{\sigma^{\eta}}{(\sigma.\eta)^2}} of size 1 ;
         \item{\code{"var_mu"}: \eqn{\sigma^{\mu}}{(\sigma.\mu)^2}} of size 1 ;
         \item{\code{"r"}: \eqn{r}{r} of size 1 ;}
         \item{\code{"phi"}: \eqn{\boldsymbol{\phi} = [\phi_1,\dots, \phi_I]^\top}{\phi = [\phi(1); ... ; \phi(I)]} of size \eqn{I} ;}
         \item{\code{"x0"}: \eqn{\mathbf{x}_0}{x(0)} of size 2 ;}
         \item{\code{"V0"}: \eqn{\mathbf{V}_0}{V(0)} of size 2 * 2 .}}}

\item{init_pars}{A list of unfitted parameters' initial values. The parameters are the same as \code{fixed_pars}.
If the user does not assign initial values for the unfitted parameters, default ones will be used.}

\item{verbose}{An integer specifying the print level of information during the algorithm (default \code{1}). Possible numbers:
\itemize{\item{\code{"0"}: no output;}
    \item{\code{"1"}: show the iteration number and \eqn{\|\Delta \boldsymbol{\Theta}_i\|}{||\Delta \Theta(i)||};}
    \item{\code{"2"}: 1 + show the obtained parameters.}}}

\item{control}{A list of control values of EM algorithm:
\itemize{\item{\code{acceleration}: TRUE/FALSE indicating whether to use the accelerated EM algorithm (default TRUE);}
   \item{\code{maxit}: Maximum number of iterations (default \code{3000});}
   \item{\code{abstol}: Absolute tolerance for parameters' change \eqn{\|\Delta \boldsymbol{\Theta}_i\|}{||\Delta \Theta(i)||} as the stopping criteria (default \code{1e-4})}
   \item{\code{log_switch}: TRUE/FALSE indicating whether to record the history of convergence progress (defalut TRUE).}}}
}
\value{
A list of class "\code{volume_model}" with the following elements (if the algorithm converges):
        \itemize{\item{\code{par}: }{A list of parameters' fitted values.}
        \item{\code{init}: }{A list of valid initial values from users.}
        \item{\code{par_log}: }{A list of intermediate parameters' values if \code{log_switch = TRUE}.}
        \item{\code{converged}: }{A list of logical values indicating whether each parameter is fitted.}
        }
}
\description{
The main function for defining and fitting a univaraite state-space model on intraday trading volume. The model is proposed in (Chen et al., 2016) as
 \deqn{\mathbf{x}_{\tau+1} = \mathbf{A}_{\tau}\mathbf{x}_{\tau} + \mathbf{w}_{\tau},}{x(\tau+1) = A(\tau) x(\tau) + w(\tau),}
             \deqn{y_{\tau} = \mathbf{C}\mathbf{x}_{\tau} + \phi_{\tau} + v_\tau,}{y(\tau) = C x(\tau) + \phi(\tau) + v(\tau),}
             where
             \itemize{\item{\eqn{\mathbf{x}_{\tau} = [\eta_{\tau}, \mu_{\tau}]^\top}{x(\tau) = [\eta(\tau); \mu(\tau)]} is the hidden state vector containing the log daily component and the log intraday dynamic component;}
                      \item{\eqn{\mathbf{A}_{\tau} = \left[\begin{array}{cc}a_{\tau}^{\eta}&0\\0&a^{\mu}\end{array} \right]}{A(\tau) = [a.\eta(\tau), 0; 0, a.\mu]}
                            is the state transition matrix with \eqn{a_{\tau}^{\eta} = \left\{\begin{array}{cl}a^{\eta} & t = kI, k = 1,2,\dots\\0 & \textrm{otherwise};\end{array}\right.}{a.\eta(\tau) = a.\eta, when \tau = kI, k = 1, 2, ... , and zero otherwise;}}
                      \item{\eqn{\mathbf{C} = [1, 1]}{C = [1, 1]} is the observation matrix;}
                      \item{\eqn{\phi_{\tau}}{\phi(\tau)} is the corresponding element from \eqn{\boldsymbol{\phi} = [\phi_1,\dots, \phi_I]^\top}{\phi = [\phi(1); ... ; \phi(I)]}, which is the log seasonal component;}
                      \item{\eqn{\mathbf{w}_{\tau} = [\epsilon_{\tau}^{\eta},\epsilon_{\tau}^{\mu}]^\top \sim \mathcal{N}(\mathbf{0}, \mathbf{Q}_{\tau})}{w(\tau) = [\epsilon.\eta(\tau); \epsilon.\mu(\tau)] ~ N(0, Q(\tau))}
                            represents the i.i.d. Gaussian noise in the state transition, with a time-varying covariance matrix
                            \eqn{\mathbf{Q}_{\tau} = \left[\begin{array}{cc}(\sigma_{\tau}^{\eta})^2&0\\ 0&(\sigma^{\mu})^2\end{array} \right]}{Q(\tau) = [(\sigma.\eta(\tau))^2, 0; 0, (\sigma.\mu)^2]}
                            and \eqn{\sigma_\tau^{\eta} = \left\{\begin{array}{cl} \sigma^{\eta} & t = kI, k = 1,2,\dots\\0 & \textrm{otherwise};\end{array}\right.}{\sigma.\eta(\tau) = \sigma.\eta, when \tau = kI, k = 1, 2, ... , and zero otherwise;}}
                       \item{\eqn{v_\tau \sim \mathcal{N}(0, r)}{v(\tau) ~ N(0, r)} is the i.i.d. Gaussian noise in the observation;}
                       \item{\eqn{\mathbf{x}_1}{x(1)} is the initial state at \eqn{\tau = 1}{\tau = 1}, and it follows \eqn{\mathcal{N}(\mathbf{x}_0, \mathbf{V}_0)}{N(x(0), V(0))}}.}
            In the model, \eqn{\boldsymbol{\Theta} = \left\{a^{\eta},a^{\mu},\sigma^{\eta},\sigma^{\mu},r,\boldsymbol{\phi}, \mathbf{x}_0, \mathbf{V}_0\right\}}{\Theta = {a.\eta, a.\mu, (\sigma.\eta)^2, (\sigma.\mu)^2, r, \phi, x(0), V(0)}}
            are treated as parameters.
The model is fitted by expectation-maximization (EM) algorithms. The implementation follows (Chen et al., 2016), and the accelerated scheme is provided in (Varadhan and Roland, 2008).
The algorithm terminates when \code{maxit} is reached or the condition \eqn{\|\Delta \boldsymbol{\Theta}_i\| \le \textrm{abstol}}{||\Delta \Theta(i)|| <= abstol} is satisfied.
}
\examples{
library(intradayModel)
data(volume_aapl)
volume_aapl_training <- volume_aapl[, 1:20]
\donttest{
# fit model with no prior knowledge
model_fit <- fit_volume(volume_aapl_training)
}
# fit model with fixed_pars and init_pars
model_fit <- fit_volume(volume_aapl_training, fixed_pars = list(a_mu = 0.5, var_mu = 0.05),
                        init_pars = list(a_eta = 0.5))
\donttest{
# fit model with other control options
model_fit <- fit_volume(volume_aapl_training, verbose = 2, 
  control = list(acceleration = FALSE, maxit = 1000, abstol = 1e-4, log_switch = FALSE))
}

}
\references{
Chen, R., Feng, Y., and Palomar, D. (2016). Forecasting intraday trading volume: A Kalman filter approach. Available at SSRN 3101695.

Varadhan, R., and Roland, C. (2008). Simple and globally convergent methods for accelerating the convergence of any EM algorithm.
Scandinavian Journal of Statistics, 35(2), 335–353.
}
\author{
Shengjie Xiu, Yifan Yu and Daniel P. Palomar
}
