#' Clean and standardize football player data
#'
#' This function converts character-based numeric fields into numeric
#' values and prepares player data for further analysis.
#'
#' @param df A data frame containing raw football player data.
#'   Must include at least columns \code{name}, \code{age},
#'   and \code{market_value_est}.
#'
#' @return A data frame with cleaned and standardized player data.
#'
#' @details
#' The function performs safe numeric conversion and does not remove
#' rows with missing values.
#'
#' @examples
#' df <- data.frame(
#'   name = c("Player A", "Player B"),
#'   age = c("21", "23"),
#'   market_value_est = c("€500k", "€750k"),
#'   club = c("Club A", "Club B"),
#'   league_country = c("Indonesia", "Indonesia"),
#'   stringsAsFactors = FALSE
#' )
#'
#' clean_player_db(df)
#'
#' @export
clean_player_db <- function(df) {

  df_clean <- df |>
    dplyr::mutate(
      # age: buang karakter aneh, ubah ke numeric
      age = stringr::str_trim(age),
      age = dplyr::na_if(age, "-"),
      age = suppressWarnings(as.numeric(age)),

      # market value: parse angka
      market_value_est = stringr::str_trim(market_value_est),
      market_value_est = dplyr::na_if(market_value_est, "-"),
      market_value_est = suppressWarnings(
        readr::parse_number(market_value_est)
      )
    )

  # JANGAN drop semuanya — hanya buang baris yang nama kosong
  df_clean |>
    dplyr::filter(!is.na(name), name != "")
}
