#include <Rcpp.h>
using namespace Rcpp;

//' @name enzyme-model
//' @param theta a vector of length 3 containing the model parameters
//' @param n the process is sampled on n equispaced points on the unit interval
//' (default 50)
//' @param E0,S0 initial state of the process
//' (default E0=100, S0=100)
//' @returns
//' `enzymeSim` returns a `n x 4' integer matrix; the ith row contains the  
//' simulated values for E, S, C and P at time t=(i-1)/(n-1).
//' @export
// [[Rcpp::export]]
IntegerMatrix enzymeSim(NumericVector theta, int n = 50, int E0 = 100,
                         int S0 = 100) {
  int i = 0, E=E0, S=S0, C=0, P=0;
  double k1 = theta[0], k2 = theta[1], k3 = theta[2], delta = 1.0 / (n - 1.0),
         tm = 0, current = 0;
  IntegerMatrix ans(n, 4);
  colnames(ans) = CharacterVector::create("E", "S", "C", "P");
  while (1) {
    double h1 = k1 * E * S, h2 = k2 * C, h3 = k3 * C, h = h1 + h2 + h3;
    current = (h < 1e-10) ? 2.0 : current + R::rexp(1 / h);
    while ((i < n) && (tm < current)) {
      ans(i, 0) = E;
      ans(i, 1) = S;
      ans(i, 2) = C;
      ans(i, 3) = P;
      tm += delta;
      i++;
    }
    if (i >= n)
      break;
    double u = R::runif(0, h);
    if (u < h1) {
      E--;
      S--;
      C++;
    } else if (u < h1 + h2) {
      C--;
      E++;
      S++;
    } else {
      C--;
      E++;
      P++;
    }
  }
  return ans;
}
