#' Michaelis-Menten enzyme kinetics
#'
#' Function `enzymeSim` (written in C++) simulates a realization from
#' the stochastic Michaelis-Menten kinetics model; function `enzymeStat`
#' computes a possible summary statistics.
#'
#' The model describes a simple case of enzyme kinetics. It involves an
#' enzyme E binding to a substrate S to form a complex C. This complex C
#' then releases a product P while simultaneously regenerating the original
#' enzyme E. The possible reactions are `E + S -> C`, `C -> E + S`, and `C -> P + E`,
#' with rates that constitute the three model parameters.
#'
#' In probabilistic terms, the integer-valued vector
#' \eqn{(E_t, S_t, C_t, P_t)'} is generated by a continuous-time Markov process
#' such that:
#' \deqn{
#' \begin{aligned}
#' &Pr\{E_{t+\delta}=E_t-1, S_{t+\delta}=S_t-1,
#' C_{t+\delta}=C_t+1, P_{t+\delta}=P_t
#' |E_t, S_t, C_t, P_t\} \\&\;\;\;\;\;\;\;\;=
#' \theta_1 E_tS_t\delta+o(\delta),  \\
#' &Pr\{E_{t+\delta}=E_t+1, S_{t+\delta}=S_t+1,
#' C_{t+\delta}=C_t-1, P_{t+\delta}=P_t
#' |E_t, S_t, C_t, P_t\} \\&\;\;\;\;\;\;\;\;=
#' \theta_2 C_t\delta+o(\delta),  \\
#' &Pr\{E_{t+\delta}=E_t+1, S_{t+\delta}=S_t,
#' C_{t+\delta}=C_t-1, P_{t+\delta}=P_t+1
#' |E_t, S_t, C_t, P_t\} \\&\;\;\;\;\;\;\;\;=
#' \theta_3 C_t\delta+o(\delta).
#' \end{aligned}
#' }
#' The initial state is \eqn{(E_0, S_0, C_0, P_0)'=(E0, S0, 0, 0)'}.
#' Process is simulated using Gillespie exact algorithm.
#' @name enzyme-model
#' @aliases enzyme
#' @references
#' Darren J. Wilkinson (2019) Stochastic Modelling for Systems Biology,
#' Third edition, CRC Press
#' @keywords model
NULL

#' @rdname enzyme-model
#' @param y
#' a `n x 4` matrix containing the observed data
#' @param degree
#' the degree of the B-splines basis (default 2)
#' @param knots
#' the knots of the B-splines basis (default: only one knot equal to 0.2)
#' @note
#' The summary statistics is based only on \eqn{C_t} and \eqn{P_t}.
#' Indeed, at every time point \eqn{t},  \eqn{E_t=E0-C_t} and
#' \eqn{S_t=S0-C_t-P_t}.
#' @return
#' `enzymeStat` returns a numeric vector of length
#' 2*(degree+1+length(knots)) containing the coefficents
#' of the B-splines curves (with degree and knots given by the second and third
#' arguments) fitted by least squares to the complex and product trajectories
#' (i.e., to the last two components of the process's state).
#' @examples
#' set.seed(1)
#' theta <- c(0.5, 2.5, 1)
#' y <- enzymeSim(theta)
#' plot(ts(y, start = 0, frequency = 50), main = "")
#' print(tobs <- enzymeStat(y))
#' \donttest{
#' # It takes some time to run
#' tsim <- function(theta) enzymeStat(enzymeSim(theta))
#' m <- ifit(tobs, tsim, l = rep(0, 3), u = rep(50, 3), trace = 1000)
#' m
#' confint(m)
#' diagIFIT(m)
#' numsimIFIT(m)
#' }
#' @export
enzymeStat <- function(y, degree = 2, knots = 0.2) {
    as.numeric(stats::coef(stats::lsfit(
        splines::bs(seq(0, 1, length = NROW(y)), degree = degree, knots = knots),
        y[, 3:4]
    )))
}
