
#' Computes the confidence interval for an ICC
#'
#' @keywords internal
#' @param icc Numeric. The intraclass correlation value.
#' @param se Numeric. The variance of the icc estimate
#' @param alpha Numeric. Significance level. Default to 0.05.
#' @param m Numeric. Number of replicates.
#' @param N Numeric. number of subjects.
#' @details
#' Confidence intervals are constructed using asymptotic methods assuming a Normal distribution.
#' The implemented methods include: the asymptotic Normal approach, Fisher's Z transformation, 
#' and the Konishi-Gupta transformation.
ic_icc<-function(icc,se,alpha=0.05, m, N){
  
  # Asymptotic
  
  ic.icc_As <- icc + c(-1,1)*qnorm(1-alpha/2)*sqrt(se)
  
  
  # Fisher Z transform
  zF <- Z_trans(icc,m)
  se.zF <- dZ(icc,m)*sqrt(se)
  ic.zF <- zF+c(-1,1)*qnorm(1-alpha/2)*se.zF
  ic.icc_ZF <- (exp(2*ic.zF)-1)/(exp(2*ic.zF)+m[1]-1)
  
  # Konishi Gupta Z transform
  zKG <- K_trans(icc,m)
  bias <-  (7-5*m)/(N*sqrt(18*m*(m-1)))
  se.zKG <- dK(icc,m)*sqrt(se)
  ic.zKG <- zKG-bias+c(-1,1)*qnorm(1-alpha/2)*se.zKG
  ic.icc_KG <- Z_inv_KG(ic.zKG,m,N)
  
  result <- matrix(c(ic.icc_As,ic.icc_ZF,ic.icc_KG),
                   ncol=2,byrow=TRUE)
  
  colnames(result)<-c("LL95%","UL95%")
  rownames(result)<-c("As.","F","KG")
  return(result)
  
  
}
