% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bpmc.R
\name{dmc}
\alias{dmc}
\title{Density of the McDonald (Mc)/Beta Power Distribution Distribution}
\usage{
dmc(x, gamma = 1, delta = 0, lambda = 1, log = FALSE)
}
\arguments{
\item{x}{Vector of quantiles (values between 0 and 1).}

\item{gamma}{Shape parameter \code{gamma} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{delta}{Shape parameter \code{delta} >= 0. Can be a scalar or a vector.
Default: 0.0.}

\item{lambda}{Shape parameter \code{lambda} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{log}{Logical; if \code{TRUE}, the logarithm of the density is
returned (\eqn{\log(f(x))}). Default: \code{FALSE}.}
}
\value{
A vector of density values (\eqn{f(x)}) or log-density values
(\eqn{\log(f(x))}). The length of the result is determined by the recycling
rule applied to the arguments (\code{x}, \code{gamma}, \code{delta},
\code{lambda}). Returns \code{0} (or \code{-Inf} if
\code{log = TRUE}) for \code{x} outside the interval (0, 1), or
\code{NaN} if parameters are invalid (e.g., \code{gamma <= 0},
\code{delta < 0}, \code{lambda <= 0}).
}
\description{
Computes the probability density function (PDF) for the McDonald (Mc)
distribution (also previously referred to as Beta Power) with parameters
\code{gamma} (\eqn{\gamma}), \code{delta} (\eqn{\delta}), and \code{lambda}
(\eqn{\lambda}). This distribution is defined on the interval (0, 1).
}
\details{
The probability density function (PDF) of the McDonald (Mc) distribution
is given by:
\deqn{
f(x; \gamma, \delta, \lambda) = \frac{\lambda}{B(\gamma,\delta+1)} x^{\gamma \lambda - 1} (1 - x^\lambda)^\delta
}
for \eqn{0 < x < 1}, where \eqn{B(a,b)} is the Beta function
(\code{\link[base]{beta}}).

The Mc distribution is a special case of the five-parameter
Generalized Kumaraswamy (GKw) distribution (\code{\link{dgkw}}) obtained
by setting the parameters \eqn{\alpha = 1} and \eqn{\beta = 1}.
It was introduced by McDonald (1984) and is related to the Generalized Beta
distribution of the first kind (GB1). When \eqn{\lambda=1}, it simplifies
to the standard Beta distribution with parameters \eqn{\gamma} and
\eqn{\delta+1}.
}
\examples{
\donttest{
# Example values
x_vals <- c(0.2, 0.5, 0.8)
gamma_par <- 2.0
delta_par <- 1.5
lambda_par <- 1.0 # Equivalent to Beta(gamma, delta+1)

# Calculate density using dmc
densities <- dmc(x_vals, gamma_par, delta_par, lambda_par)
print(densities)
# Compare with Beta density
print(stats::dbeta(x_vals, shape1 = gamma_par, shape2 = delta_par + 1))

# Calculate log-density
log_densities <- dmc(x_vals, gamma_par, delta_par, lambda_par, log = TRUE)
print(log_densities)

# Compare with dgkw setting alpha = 1, beta = 1
densities_gkw <- dgkw(x_vals,
  alpha = 1.0, beta = 1.0, gamma = gamma_par,
  delta = delta_par, lambda = lambda_par
)
print(paste("Max difference:", max(abs(densities - densities_gkw)))) # Should be near zero

# Plot the density for different lambda values
curve_x <- seq(0.01, 0.99, length.out = 200)
curve_y1 <- dmc(curve_x, gamma = 2, delta = 3, lambda = 0.5)
curve_y2 <- dmc(curve_x, gamma = 2, delta = 3, lambda = 1.0) # Beta(2, 4)
curve_y3 <- dmc(curve_x, gamma = 2, delta = 3, lambda = 2.0)

plot(curve_x, curve_y2,
  type = "l", main = "McDonald (Mc) Density (gamma=2, delta=3)",
  xlab = "x", ylab = "f(x)", col = "red", ylim = range(0, curve_y1, curve_y2, curve_y3)
)
lines(curve_x, curve_y1, col = "blue")
lines(curve_x, curve_y3, col = "green")
legend("topright",
  legend = c("lambda=0.5", "lambda=1.0 (Beta)", "lambda=2.0"),
  col = c("blue", "red", "green"), lty = 1, bty = "n"
)
}

}
\references{
McDonald, J. B. (1984). Some generalized functions for the size distribution
of income. \emph{Econometrica}, 52(3), 647-663.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{dgkw}} (parent distribution density),
\code{\link{pmc}}, \code{\link{qmc}}, \code{\link{rmc}} (other Mc functions),
\code{\link[stats]{dbeta}}
}
\author{
Lopes, J. E.
}
\keyword{density}
\keyword{distribution}
\keyword{mcdonald}
