#' Print function for gapclosing objects
#' @description Prints the same output as generated by \code{summary}
#' @param x An object of class \code{gapclosing}, which results from a call to the function \code{gapclosing}
#' @param ... Other arguments to \code{print} commands
#' @param digits Argument passed to print.data.frame
#' @param quote Argument passed to print.data.frame
#' @param right Argument passed to print.data.frame
#' @param row.names Argument passed to print.data.frame
#' @param max Argument passed to print.data.frame
#' @return Prints a summary of the estimates.
#' @references Lundberg I (2021). "The gap-closing estimand: A causal approach to study interventions that close disparities across social categories." Sociological Methods and Research. Available at {https://osf.io/gx4y3/}.
#' @importFrom magrittr %>%
#' @export

print.gapclosing <- function(
  x,
  ...,
  digits = 2,
  quote = FALSE,
  right = FALSE,
  row.names = FALSE,
  max = NULL
) {

  # Prepare for non-standard evaluation
  change_type <- change_formula <- NULL

  cat("\nFactual mean outcomes:\n")
  print(x$factual_means,
        digits = digits,
        quote = quote,
        right = right,
        row.names = row.names,
        max = max, scientific = FALSE)
  cat("\nCounterfactual mean outcomes (post-intervention means):\n")
  print(x$counterfactual_means,
        digits = digits,
        quote = quote,
        right = right,
        row.names = row.names,
        max = max, scientific = FALSE)
  cat("\nFactual disparities:\n")
  print(x$factual_disparities,
        digits = digits,
        quote = quote,
        right = right,
        row.names = row.names,
        max = max, scientific = FALSE)
  cat("\nCounterfactual disparities (gap-closing estimands):\n")
  print(x$counterfactual_disparities,
        digits = digits,
        quote = quote,
        right = right,
        row.names = row.names,
        max = max, scientific = FALSE)
  cat("\nAdditive gap closed: Counterfactual - Factual\n")
  print(x$change_disparities %>%
          dplyr::filter(change_type == "additive") %>%
          dplyr::select(-change_type, -change_formula),
        digits = digits,
        quote = quote,
        right = right,
        row.names = row.names,
        max = max, scientific = FALSE)
  cat("\nProportional gap closed: (Counterfactual - Factual) / Factual\n")
  print(x$change_disparities %>%
          dplyr::filter(change_type == "proportional") %>%
          dplyr::select(-change_type, -change_formula),
        digits = digits,
        quote = quote,
        right = right,
        row.names = row.names,
        max = max, scientific = FALSE)
}
