#' Generic Plot Function
#'
#' This is a generic `plot` function that dispatches to specific `plot` methods
#' based on the class of the object provided. It is used to create plots for objects
#' such as `fuzzy_lm`.
#'
#' @param object The object to be plotted.
#' @param ... Additional arguments passed to specific plot methods.
#'
#' @return Depends on the class of `object`. Typically, a plot or visualization is returned.
#' @examples
#' # Example with fuzzy_lm:
#' set.seed(123)
#' x_crisp <- seq(4, 12, length.out = 20)
#' beta <- 1.5
#' intercept <- 2
#' y_crisp <- intercept + beta * x_crisp + rnorm(length(x_crisp), mean = 0, sd = 0.5)
#'
#' # Fuzzify data
#' spread_x <- 0.5
#' spread_y <- 1.0
#' X_fuzzy <- fuzzify_crisp_matrix(matrix(x_crisp, ncol = 1), spread = spread_x)
#' Y_fuzzy <- fuzzify_crisp_vector(y_crisp, spread = spread_y)
#'
#' # Fit fuzzy regression model
#' object <- fuzzy_lm(X_fuzzy, Y_fuzzy, p = 1)
#'
#' # Plot
#' \donttest{
#' plot(object, X_fuzzy = X_fuzzy, Y_fuzzy = Y_fuzzy)
#' }
#' @export
plot <- function(object, ...) {
  UseMethod("plot")
}

#' Plot Fuzzy Regression Results
#'
#' Visualizes the results of a fuzzy regression model. For simple regression
#' (1 predictor), it generates a 2D plot with fuzzy intervals and regression lines.
#' For multiple regression (2 predictors), it generates a 3D plot with cubes representing
#' fuzzy intervals and a regression plane.
#'
#' @param object An object of class `fuzzy_lm`.
#' @param ... Additional arguments passed to the method, including:
#'   \itemize{
#'     \item `X_fuzzy`: A list of fuzzified predictor variables.
#'     \item `Y_fuzzy`: A list of fuzzified outcome variables.
#'   }
#'
#' @return A `ggplot2` object for simple regression or a `plotly` object for multiple regression.
#' @examples
#' # Example 1: Simple Regression
#' # See above for setup example
#'
#' # Example 2: Multiple Regression
#' set.seed(123)
#' n <- 100
#' x1_crisp <- runif(n, 5, 15)
#' x2_crisp <- runif(n, 10, 20)
#' beta <- c(3, 1.5, -0.8)
#' y_crisp <- beta[1] + beta[2] * x1_crisp + beta[3] * x2_crisp + rnorm(n, mean = 0, sd = 2)
#'
#' X_fuzzy <- fuzzify_crisp_matrix(cbind(x1_crisp, x2_crisp), spread = 0.5)
#' Y_fuzzy <- fuzzify_crisp_vector(y_crisp, spread = 1.0)
#' object <- fuzzy_lm(X_fuzzy, Y_fuzzy, p = 2)
#' \donttest{
#' plot(object, X_fuzzy = X_fuzzy, Y_fuzzy = Y_fuzzy)
#' }
#' @importFrom ggplot2 ggplot geom_rect aes geom_point geom_abline labs theme_minimal
#' @importFrom plotly plot_ly add_trace add_markers add_surface layout
#' @importFrom rlang .data
#' @importFrom magrittr %>%
#' @export
#' @method plot fuzzy_lm
plot.fuzzy_lm <- function(object, ...) {
  stopifnot(inherits(object, "fuzzy_lm"))

  # Extract additional arguments
  args <- list(...)
  if (!"X_fuzzy" %in% names(args) || !"Y_fuzzy" %in% names(args)) {
    stop("Both 'X_fuzzy' and 'Y_fuzzy' must be provided via the '...' arguments.")
  }
  X_fuzzy <- args$X_fuzzy
  Y_fuzzy <- args$Y_fuzzy

  # Extract necessary components
  Coefficients <- object$Coefficients
  num_predictors <- nrow(Coefficients) - 1  # Number of predictors

  if (num_predictors == 1) {
    # Simple regression (2D)
    fuzzy_data <- data.frame(
      x_lower = sapply(X_fuzzy, function(x) x[[1]]$l),
      x_middle = sapply(X_fuzzy, function(x) x[[1]]$x),
      x_upper = sapply(X_fuzzy, function(x) x[[1]]$r),
      y_lower = sapply(Y_fuzzy, function(y) y$l),
      y_middle = sapply(Y_fuzzy, function(y) y$x),
      y_upper = sapply(Y_fuzzy, function(y) y$r)
    )

    ggplot2::ggplot(fuzzy_data) +
      ggplot2::geom_rect(ggplot2::aes(
        xmin = .data$x_lower, xmax = .data$x_upper,
        ymin = .data$y_lower, ymax = .data$y_upper
      ), fill = "lightblue", alpha = 0.4) +
      ggplot2::geom_point(ggplot2::aes(
        x = .data$x_middle, y = .data$y_middle
      ), size = 2, color = "blue") +
      ggplot2::geom_abline(
        intercept = Coefficients[1, "Estimate"],
        slope = Coefficients[2, "Estimate"],
        color = "red", linetype = "dashed", linewidth = 1
      ) +
      ggplot2::labs(
        title = "Fuzzy Regression (2D)",
        x = "Fuzzy Predictor (X)",
        y = "Fuzzy Outcome (Y)"
      ) +
      ggplot2::theme_minimal()
  } else if (num_predictors == 2) {
    # Multiple regression (3D)
    plot_data <- data.frame(
      x1_l = sapply(X_fuzzy, function(fz) fz[[1]]$l),
      x1_u = sapply(X_fuzzy, function(fz) fz[[1]]$r),
      x2_l = sapply(X_fuzzy, function(fz) fz[[2]]$l),
      x2_u = sapply(X_fuzzy, function(fz) fz[[2]]$r),
      y_l = sapply(Y_fuzzy, function(fz) fz$l),
      y_u = sapply(Y_fuzzy, function(fz) fz$r),
      x1 = sapply(X_fuzzy, function(fz) fz[[1]]$x),
      x2 = sapply(X_fuzzy, function(fz) fz[[2]]$x),
      y = sapply(Y_fuzzy, function(fz) fz$x)
    )

    x1_seq <- seq(min(plot_data$x1), max(plot_data$x1), length.out = 10)
    x2_seq <- seq(min(plot_data$x2), max(plot_data$x2), length.out = 10)
    grid <- expand.grid(x1 = x1_seq, x2 = x2_seq)
    grid$z <- Coefficients[1, "Estimate"] +
      Coefficients[2, "Estimate"] * grid$x1 +
      Coefficients[3, "Estimate"] * grid$x2

    edge_lines <- lapply(1:nrow(plot_data), function(i) {
      with(plot_data[i, ], {
        list(
          list(x = c(x1_l, x1_u), y = c(x2_l, x2_l), z = c(y_l, y_l)),
          list(x = c(x1_u, x1_u), y = c(x2_l, x2_u), z = c(y_l, y_l)),
          list(x = c(x1_u, x1_l), y = c(x2_u, x2_u), z = c(y_l, y_l)),
          list(x = c(x1_l, x1_l), y = c(x2_u, x2_l), z = c(y_l, y_l)),
          list(x = c(x1_l, x1_u), y = c(x2_l, x2_l), z = c(y_u, y_u)),
          list(x = c(x1_u, x1_u), y = c(x2_l, x2_u), z = c(y_u, y_u)),
          list(x = c(x1_u, x1_l), y = c(x2_u, x2_u), z = c(y_u, y_u)),
          list(x = c(x1_l, x1_l), y = c(x2_u, x2_l), z = c(y_u, y_u)),
          list(x = c(x1_l, x1_l), y = c(x2_l, x2_l), z = c(y_l, y_u)),
          list(x = c(x1_u, x1_u), y = c(x2_l, x2_l), z = c(y_l, y_u)),
          list(x = c(x1_u, x1_u), y = c(x2_u, x2_u), z = c(y_l, y_u)),
          list(x = c(x1_l, x1_l), y = c(x2_u, x2_u), z = c(y_l, y_u))
        )
      })
    })

    edge_lines <- do.call(c, edge_lines)
    fig <- plotly::plot_ly()

    for (edge in edge_lines) {
      fig <- fig %>%
        plotly::add_trace(
          x = edge$x, y = edge$y, z = edge$z,
          type = "scatter3d", mode = "lines",
          line = list(color = "black", width = 1),
          showlegend = FALSE
        )
    }

    fig <- fig %>%
      plotly::add_markers(
        x = plot_data$x1, y = plot_data$x2, z = plot_data$y,
        marker = list(size = 5, color = "blue"),
        showlegend = FALSE
      ) %>%
      plotly::add_surface(
        x = matrix(x1_seq, nrow = 10, ncol = 10),
        y = matrix(x2_seq, nrow = 10, ncol = 10, byrow = TRUE),
        z = matrix(grid$z, nrow = 10, ncol = 10),
        showscale = FALSE,
        opacity = 0.6
      ) %>%
      plotly::layout(
        scene = list(
          xaxis = list(title = "Predictor X1"),
          yaxis = list(title = "Predictor X2"),
          zaxis = list(title = "Outcome Y")
        )
      )

    return(fig)
  } else {
    stop("This plotting function supports only simple (1 predictor) or multiple (2 predictors) regression.")
  }
}
