% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PASS_Proj_Test_ufDA.R
\name{PASS_Proj_Test_ufDA}
\alias{PASS_Proj_Test_ufDA}
\title{Power and Sample size (PASS) calculation of
Two-Sample Projection-based test for sparsely observed univariate functional data.}
\usage{
PASS_Proj_Test_ufDA(
  sample_size,
  target.power,
  sig.level = 0.05,
  nobs_per_subj,
  obs.design,
  mean_diff_fnm,
  cov.type = c("ST", "NS"),
  cov.par,
  sigma2.e,
  missing_type = c("nomiss", "constant"),
  missing_percent = 0,
  eval_SS = 5000,
  alloc.ratio = c(1, 1),
  fpca_method = c("fpca.sc", "face"),
  mean_diff_add_args = list(),
  fpca_optns = list(pve = 0.95),
  nWgrid = 201,
  npc_to_use = NULL,
  return.eigencomp = FALSE,
  nsim = 10000
)
}
\arguments{
\item{sample_size}{Total sample size combining both the groups, must be a positive integer.}

\item{target.power}{Target power to achieve, must be a number between 0 and 1.
Only one of \code{sample_size} and \code{target.power} should be non-null. The
function will return sample size if \code{sample_size} is NULL, and
return power if \code{target.power} is NULL.}

\item{sig.level}{Significance level of the test, default set at 0.05, must be less than 0.2.}

\item{nobs_per_subj}{The number of observations per subject. Each element of it must be greater than 3.
It could also be a vector to indicate that the number of observation for each is randomly varying
between the elements of the vector, or a scalar to ensure that the number of observations are same
for each subject. See examples.}

\item{obs.design}{The sampling design of the observations. Must be provided as
a list with the following elements. If the design is longitudinal (e.g. a clinical trial
where there is pre-specified schedule of visit for the participants) it must be
a named list with elements \code{design}, \code{visit.schedule} and \code{visit.window}, where
\code{obs.design$design} must be specified as \code{'longitudinal'}, \code{visit.schedule}
specifying schedule of visits (in months or days or any unit of time), other than the baseline visit
and \code{visit.window} denoting the maximum time window for every visit.
For functional design (where the observation points are either densely observed within a
compact interval or under a sparse random design), the argument must be provided
as a named list with elements \code{design} and \code{fun.domain}, where
\code{obs.design$design} must be specified as \code{'functional'} and \code{obs.design$fun.domain}
must be specified as a two length vector indicating the domain of the function.
See Details on the specification of arguments section below more details.}

\item{mean_diff_fnm}{The name of the function that output of the difference of the mean between the
two groups at any given time. It must be supplied as character, so that \code{match.fun(mean_diff_fnm)}
returns a valid function, that takes a vector input, and returns a vector of the same length of the input.}

\item{cov.type}{The type of the covariance structure of the data, must be either of 'ST' (stationary) or
'NS' (non-stationary). This argument along with the \code{cov.par} argument must be
specified compatibly to ensure that the function does not return an error. See the details
of \code{cov.par} argument.}

\item{cov.par}{The covariance structure of the latent response trajectory.
If \code{cov.type == 'ST'} then, \code{cov.par}
must be specified a named list of two elements, \code{var} and \code{cor},
where \code{var} is the common variance of the observations, which must be a
positive number; and \code{cor} specifies the correlation structure between
the observations. \code{cov.par$cor} must be specified in the form of the
\link[nlme:corClasses]{nlme::corClasses} specified in R package \pkg{nlme}.
Check the package documentation for more details for each of the correlation classes.
The \code{cov.par$cor} must be a \code{corStruct} class so it can be
passed onto the \code{\link[nlme:corMatrix]{nlme::corMatrix()}} to extract the subject-specific covariance matrix.
If \code{cov.type='NS'} then, \code{cov.par}
must be a named list of two elements, \code{cov.obj} and \code{eigen.comp},
where only one of the \code{cov.par$cov.obj} or \code{cov.par$eigen.comp}
must be non-null. This is to specify that the covariance structure of the
latent trajectory can be either provided in the form of covariance function or
in the form of eigenfunction and eigenvalues (Spectral decomposition).
If the \code{cov.par$cov.obj} is specified, then it must be a bivariate function,
with two arguments. Alternatively, if the true eigenfunctions  are known,
then the user can specify that by specifying \code{cov.par$eigen.comp}.
In this case, the \code{cov.par$eigen.comp} must be a named list with two elements,
\code{eig.obj} and \code{eig.val}, where \code{cov.par$eigen.comp$eig.val}
must be positive vector and \code{cov.par$eigen.comp$eig.obj}
must be a vectorized function so that its evaluation at a vector of time points
returns a matrix of dimension r by \code{length(cov.par$eigen.comp$eig.val)},
with r being the length of time points.}

\item{sigma2.e}{Measurement error variance, should be set as zero or a very small number
if the measurement error is not significant.}

\item{missing_type}{The type of missing in the number of observations of the subjects. Can be one of
\code{'nomiss'} for no missing observations
or \code{'constant'} for constant
missing percentage at every time point. The current version of package only supports
\code{missing_type = 'constant'}.}

\item{missing_percent}{The percentage of missing at each observation points for each subject.
Must be supplied as number between [0, 0.8], as missing percentage more than 80\% is not practical.
If \code{nobs_per_subj} is supplied as vector, then \code{missing_type}
is forced to set as \code{'nomiss'} and \code{missing_percent = 0}, because
the \code{missing_type = 'constant'} has no meaning if the number of observations are
varying between the subject at the first, typically considered in
the case of sparse random functional design.}

\item{eval_SS}{The sample size based on which the eigencomponents will be estimated from data.
To compute the theoretical power of the test we must make sure that we use a large enough sample size
to generate the data such that the estimated eigenfunctions are very close to the true eigenfunctions
and that the sampling design will not have much effect on the loss of precision. Default value 5000.}

\item{alloc.ratio}{The allocation ratio of samples in the each group. Note that the eigenfunctions
will still be estimated based on the total sample_size, however, the variance
of the \code{shrinkage} scores (which is required to compute the power function) will be
estimated based on the allocation of the samples in each group. Must be given as vector of
length 2. Default value is set at \code{c(1, 1)}, indicating equal sample size.}

\item{fpca_method}{The method by which the FPCA is computed. Must be one of
'fpca.sc' and 'face'. If \code{fpca_method == 'fpca.sc'} then the eigencomponents
are estimated using the function \code{\link[refund:fpca.sc]{refund::fpca.sc()}}. However, since the \code{\link[refund:fpca.sc]{refund::fpca.sc()}}
function fails to estimate the correct \code{shrinkage} scores, and throws \code{NA} values
when the measurement errors is estimated to be zero, we wrote out a similar function
where we corrected those error in current version of \code{\link[refund:fpca.sc]{refund::fpca.sc()}}. Check out
the \code{\link[=fpca_sc]{fpca_sc()}} function for details. If \code{fpca_method == 'face'}, then
the eigencomponents are estimated using \code{\link[face:face.sparse]{face::face.sparse()}} function.}

\item{mean_diff_add_args}{Additional arguments to be passed to group difference
function specified in the argument \code{mean_diff_fnm}.}

\item{fpca_optns}{Additional options to be passed onto either of \code{\link[=fpca_sc]{fpca_sc()}}
or \code{\link[face:face.sparse]{face::face.sparse()}} function in order
to estimate the eigencomponents. It must be a named list with elements
to be passed onto the respective function, depending on the \code{fpca_method}.
The names of the list must not match either of
\code{c('data', 'newdata', 'argvals.new')}
for \code{fpca_method == 'face'} and must not match either of
\code{c('ydata', 'Y.pred')} for  \code{fpca_method == 'fpca.sc'}.}

\item{nWgrid}{The length of the working grid based in the domain of the function on which
the eigenfunctions will be estimated. The actual working grid will be calculated using
the \code{\link[gss:gauss.quad]{gss::gauss.quad()}} function (so that it facilitates the numerical integration of
the eigenfunction with the mean function using gaussian quadrature rule)}

\item{npc_to_use}{Number of eigenfunctions to use to compute the power. Default is NULL, in
which case all the eigenfunctions estimated from the data will be used.}

\item{return.eigencomp}{Indicates whether to return the eigencomponents obtained from the fPCA
on the large data with sample size equal to \code{eval_SS}. Default is \code{FALSE}.}

\item{nsim}{The number of samples to be generated from the alternate distribution of
Hotelling T statistic. Default value is 10000.}
}
\value{
A list with following elements, \code{power_value} if \code{is.null(target.power)}
then returns the power of the test when n equal to \code{sample_size}, otherwise \code{required_SS},
the sample size required to achieve the power of the test at \code{target.power}.
If \code{return.eigencomp == TRUE} then \code{est_eigencomp} is also returned, containing
the entire output obtained from internal call of \code{\link[=Extract_Eigencomp_fDA]{Extract_Eigencomp_fDA()}}.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

The function \code{PASS_Proj_Test_ufDA()} computes the power and sample size (PASS) required to conduct
the projection-based test of mean function between two groups of longitudinal data
or sparsely observed functional data under a random irregular design, under
common covariance structure between the groups. See Wang (2021) for more details
of the testing procedure.
}
\details{
The function is designed to perform the power and sample size analysis
for functional under a dense and sparse (random) design and longitudinal data. The
function can handle data from wide variety of covariance structure, can be parametric,
or non-parametric. Additional with traditional stationary structures assumed for longitudinal
data (see \link[nlme:corClasses]{nlme::corClasses}), the user can specify any other non-stationary covariance
function in the form of either a covariance function or in terms of eigenfunctions and
eigenvalues. The user have a lot of flexibility into tweaking the arguments of the function
to assess the power function of the test under different sampling design and covariance process
of the response trajectory, and for any arbitrary mean difference function. Overall, the
functionality of the module is quite comprehensive and includes all the different cases
considered in the 'NCSS PASS (2023)' software. We believe that this software can be an effective
clinical trial design tools when considering the projection-based test as the primary
decision making method.
}
\section{Specification of key arguments}{


If \code{obs.design$design == 'functional'} then a dense grid of length,
specified by ngrid (typically 101/201) is internally created, and
the observation points will be randomly chosen from them.
The time points could also randomly chosen between
any number between the interval, but then for large number of subject,
\code{\link[=fpca_sc]{fpca_sc()}} function will take huge
time to estimate the eigenfunction. For dense design, the user must set
a large value of the argument \code{nobs_per_subj} and for sparse (random) design,
\code{nobs_per_subj} should be set small (and varying).
On the other hand, typical to longitudinal data, if the measurements are
taken at fixed time points (from baseline)
for each subject, then the user must set \code{obs.design$design == 'longitudinal'} and
the time points must be accordingly specified
in the argument \code{obs.design$visit.schedule}. The length of \code{obs.design$visit.schedule}
must match \code{length(nobs_per_subj)-1}. Internally, when
\code{obs.design$design == 'longitudinal'}, the function scale the visit times
so that it lies between [0, 1], so the user should not
specify any element named \code{fun.domain} in the
list for \code{obs.design$design == 'longitudinal'}. Make sure that
the mean function and the covariance function specified
in the \code{cov.par} and \code{mean_diff_fnm} parameter also scaled to
take argument between [0, 1]. Also, it is imperative to say that \code{nobs_per_subj} must
be of a scalar positive integer for \code{design == 'longitudinal'}.
}

\examples{

# Example 1: Power analysis for stationary exponential covariance.
# Should return a power same as the size because
# the true mean difference is zero.

set.seed(12345)
mean.diff <- function(t) {0*t};
obs.design = list("design" = "longitudinal",
                  "visit.schedule" = seq(0.1, 0.9, length.out=7),
                   "visit.window" = 0.05)
cor.str <- nlme::corExp(1, form = ~ time | Subject);
sigma2 <- 1; sigma2.e <- 0.25; nobs_per_subj <- 8;
missing_type <- "constant"; missing_percent <- 0.01;
# Please increase `eval_SS` argument from 1000 to 5000 to get
# accurate precision on the estimated eigenfunctions.
pow  <- PASS_Proj_Test_ufDA(sample_size = 100, target.power = NULL, sig.level = 0.05,
                            obs.design = obs.design,
                            mean_diff_fnm = "mean.diff", cov.type = "ST",
                            cov.par = list("var" = sigma2, "cor" = cor.str),
                            sigma2.e = sigma2.e, nobs_per_subj = nobs_per_subj,
                            missing_type = missing_type,
                            missing_percent = missing_percent, eval_SS = 1000,
                            alloc.ratio = c(1,1), nWgrid = 201,
                            fpca_method = "fpca.sc",
                            mean_diff_add_args = list(), fpca_optns = list("pve" = 0.95),
                            nsim = 1e3)

print(pow$power_value)

# Example 2: Sample size calculation for a non-stationary covariance:

alloc.ratio  <- c(1,1)
mean.diff    <- function(t) {3 * (t^3)};
eig.fun <- function(t, k) {
  if (k==1) ef <- sqrt(2)*sin(2*pi*t)
  else if (k==2) ef <- sqrt(2)*cos(2*pi*t)
  return(ef)}
eig.fun.vec  <- function(t){cbind(eig.fun(t, 1),eig.fun(t, 2))}
eigen.comp   <- list("eig.val" = c(1, 0.5), "eig.obj" = eig.fun.vec)
obs.design   <- list(design = "functional", fun.domain = c(0,1))
cov.par      <- list("cov.obj" = NULL, "eigen.comp" = eigen.comp)
sigma2.e     <- 0.001; nobs_per_subj <- 4:7;
missing_type <- "nomiss"; missing_percent <- 0;
fpca_method  <- "fpca.sc"
# Please increase `eval_SS` argument from 1000 to 5000 to get
# accurate precision on the estimated eigenfunctions.
pow  <- PASS_Proj_Test_ufDA(sample_size = NULL, target.power = 0.8,
                            sig.level = 0.05, obs.design = obs.design,
                            mean_diff_fnm = "mean.diff", cov.type = "NS",
                            cov.par = cov.par, sigma2.e = sigma2.e,
                            nobs_per_subj = nobs_per_subj, missing_type = missing_type,
                            missing_percent = missing_percent, eval_SS = 1000,
                            alloc.ratio = alloc.ratio, fpca_method = "fpca.sc",
                            mean_diff_add_args = list(), fpca_optns = list(pve = 0.95),
                            nsim = 1e3, nWgrid = 201)

print(pow$required_SS)

}
\references{
Wang, Qiyao (2021)
\emph{Two-sample inference for sparse functional data,  Electronic Journal of Statistics,
Vol. 15, 1395-1423}
\doi{https://doi.org/10.1214/21-EJS1802}. \cr \cr
PASS 2023 Power Analysis and Sample Size Software (2023). NCSS, LLC. Kaysville, Utah, USA, ncss.com/software/pass.
}
\seealso{
See \code{\link[=Power_Proj_Test_ufDA]{Power_Proj_Test_ufDA()}} and \code{\link[=Extract_Eigencomp_fDA]{Extract_Eigencomp_fDA()}}.
}
\author{
Salil Koner \cr Maintainer: Salil Koner
\email{salil.koner@duke.edu}
}
