% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core.R
\name{sma}
\alias{sma}
\title{Sparse Matrix Approximation}
\usage{
sma(
  x,
  k = min(5, dim(x)),
  gamma = NULL,
  rotate = c("varimax", "absmin"),
  shrink = c("soft", "hard"),
  center = FALSE,
  scale = FALSE,
  normalize = FALSE,
  order = FALSE,
  flip = FALSE,
  max.iter = 1000,
  epsilon = 1e-05,
  quiet = TRUE
)
}
\arguments{
\item{x}{\code{matrix} or \code{Matrix} to be analyzed.}

\item{k}{\code{integer}, rank of approximation.}

\item{gamma}{\code{numeric(2)}, sparsity parameters. If \code{gamma} is \code{numeric(1)}, it is used for both left and right sparsity component (i.e, \code{z} and \code{y}). If absent, the two parameters are set as (default): \code{sqrt(nk)} and \code{sqrt(pk)} for \code{z} and \code{y} respectively, where n x p is the dimension of \code{x}.}

\item{rotate}{\code{character(1)}, rotation method. Two options are currently available: "varimax" (default) or "absmin" (see details).}

\item{shrink}{\code{character(1)}, shrinkage method, either "soft"- (default) or "hard"-thresholding (see details).}

\item{center}{\code{logical}, whether to center columns of \code{x} (see \code{\link[=scale]{scale()}}).}

\item{scale}{\code{logical}, whether to scale columns of \code{x} (see \code{\link[=scale]{scale()}}).}

\item{normalize}{\code{logical}, whether to rows normalization should be done before and undone afterward the rotation (see details).}

\item{order}{\code{logical}, whether to re-order the columns of the estimates (see Details below).}

\item{flip}{\code{logical}, whether to flip the signs of the columns of estimates such that all columns are positive-skewed (see details).}

\item{max.iter}{\code{integer}, maximum number of iteration (default to 1,000).}

\item{epsilon}{\code{numeric}, tolerance of convergence precision (default to 0.00001).}

\item{quiet}{\code{logical}, whether to mute the process report (default to \code{TRUE})}
}
\value{
an \code{sma} object that contains:
\item{\code{z}, \code{b}, \code{t(y)}}{the three parts in the SMA.
\code{z} is a sparse n x k \code{matrix} that contains the row components (loadings).
The row names of \code{z} inherit the row names of \code{x}.
\code{b} is a k x k \code{matrix} that contains the scores of SMA;
the Frobenius norm of \code{b} equals to the total variance explained by the SMA.
\code{y} is a sparse n x k \code{matrix}that contains the column components (loadings).}
The row names of \code{y} inherit the column names of \code{x}.
\item{score}{the total variance explained by the SMA.
This is the optimal objective value obtained.}
\item{n.iter}{\code{integer}, the number of iteration taken.}
}
\description{
Perform the sparse matrix approximation (SMA) of a data matrix \code{x} as three multiplicative components: \code{z}, \code{b}, and \code{t(y)},
where \code{z} and \code{y} are sparse, and \code{b} is low-rank but not necessarily diagonal.
}
\details{
\strong{\code{rotate}}: The \code{rotate} option specifies the rotation technique to
use. Currently, there are two build-in options—“varimax” and “absmin”.
The “varimax” rotation maximizes the element-wise L4 norm of the rotated
matrix. It is faster and computationally more stable. The “absmin”
rotation minimizes the absolute sum of the rotated matrix. It is sharper
(as it directly minimizes the L1 norm) but slower and computationally
less stable.

\strong{\code{shrink}}: The \code{shrink} option specifies the shrinkage operator to
use. Currently, there are two build-in options—“soft”- and
“hard”-thresholding. The “soft”-thresholding universally reduce all
elements and sets the small elements to zeros. The “hard”-thresholding
only sets the small elements to zeros.

\strong{\code{normalize}}: The argument \code{normalize} gives an indication of if and
how any normalization should be done before rotation, and then undone
after rotation. If normalize is \code{FALSE} (the default) no normalization
is done. If normalize is \code{TRUE} then Kaiser normalization is done. (So
squared row entries of normalized \code{x} sum to 1.0. This is sometimes
called Horst normalization.) For \code{rotate="absmin"}, if \code{normalize} is a
vector of length equal to the number of indicators (i.e., the number of
rows of \code{x}), then the columns are divided by \code{normalize} before
rotation and multiplied by \code{normalize} after rotation. Also, If
\code{normalize} is a function then it should take \code{x} as an argument and
return a vector which is used like the vector above.

\strong{\code{order}}: In PCA (and SVD), the principal components (and the
singular vectors) are ordered. For this, we order the sparse components
(i.e., the columns of \code{z} or \code{y}) by their explained variance in the
data, which is defined as \code{sum((x \%*\% y)^2)}, where y is a column of the
sparse component. Note: not to be confused with the cumulative
proportion of variance explained by \code{y} (and \code{z}), particularly when \code{y}
(and \code{z}) is may not be strictly orthogonal.

\strong{\code{flip}}: The argument \code{flip} gives an indication of if and the
columns of estimated sparse component should be flipped. Note that the
estimated (sparse) loadings, i.e., the weights on original variables,
are column-wise invariant to a sign flipping. This is because flipping
of a principal direction does not influence the amount of the explained
variance by the component. If \code{flip=TRUE}, then the columns of loadings
will be flip accordingly, such that each column is positive-skewed. This
means that for each column, the sum of cubic elements (i.e., \code{sum(x^3)})
are non-negative.
}
\examples{
## simulate a rank-5 data matrix with some additive Gaussian noise
n <- 300
p <- 50
k <- 5 ## rank
z <- shrinkage(polar(matrix(runif(n * k), n, k)), sqrt(n))
b <- diag(5) * 3
y <- shrinkage(polar(matrix(runif(p * k), p, k)), sqrt(p))
e <- matrix(rnorm(n * p, sd = .01), n, p)
x <- scale(z \%*\% b \%*\% t(y) + e)

## perform sparse matrix approximation
s.sma <- sma(x, k)
s.sma

}
\references{
Chen, F. and Rohe, K. (2020) "A New Basis for Sparse Principal Component Analysis."
}
\seealso{
\link{sca}, \link{prs}
}
