#' Calculate the change in activation for a specific cue or set of cues. 
#' 
#' @description Calculate the change in activation for a specific cue 
#' or set of cues for all outcomes (or a subset) in the weightmatrices.
#' 
#' @export
#' @param wmlist A list with weightmatrices, generated by 
#' \code{\link{RWlearning}} or \code{\link{updateWeights}}, 
#' or a single weightmatrix (matrix). 
#' @param cueset String, specifying the cue set for which to calculate 
#' change in activation.
#' @param split String, separator between cues and/or outcomes.
#' @param select.outcomes Optional selection of outcomes to limit 
#' (or expand) the number of activations that are returned. 
#' The value of NULL (default) will 
#' return all activations (for each outcome in \code{wmlist}). 
#' Note that specified values that are not in 
#' the weightmatrix will return the initial value without error or 
#' warning. Please use \code{\link{getValues}} for returning all 
#' outcomes in the data.
#' @param init.value Value of activations for non-existing connections. 
#' Typically set to 0.
#' @param normalize Logical: whether or not the activation is normalized by 
#' dividing the total activation by the number of cues. Default is FALSE. If 
#' set to TRUE, the activation reflects the average activation per cue.
#' @return List of data frames. 
#' For each cueset defined in \code{cueset}, a dataframe of 
#' activation values is provided. These are returned as a list, with the 
#' cuesets as names. 
#' @section Notes:
#' The outcomes are selected based on the weightmatrices, and not 
#' necessarily all outcomes present in the training data. 
#' @author Jacolien van Rij
#' @seealso \code{\link{getWeightsByCue}},
#' \code{\link{getWeightsByOutcome}}
#' @family functions for calculating activations
#' @examples
#' # load example data:
#' data(dat)
#' 
#' # add obligatory columns Cues, Outcomes, and Frequency:
#' dat$Cues <- paste("BG", dat$Shape, dat$Color, sep="_")
#' dat$Outcomes <- dat$Category
#' dat$Frequency <- dat$Frequency1
#' head(dat)
#' 
#' # now use createTrainingData to sample from the specified frequencies: 
#' train <- createTrainingData(dat)
#' head(train)
#' 
#' # this training data can actually be used train network:
#' wm <- RWlearning(train)
#' 
#' # Now we calculate the activations for all outcomes
#' # per event:
#' activations <- activationsCueSet(wm, cueset="BG_bicycle_red")
#' names(activations)
#' head(activations[[1]])
#' 
#' # plot:
#' a1 <- activations[[1]]
#' emptyPlot(nrow(a1), range(a1),
#'     xlab="Learning events", ylab="Activations",
#'     xmark=TRUE, ymark=TRUE, las=1)
#' for(i in 1:ncol(a1)){
#'     lines(a1[,i], col=i, lty=i)
#' }
#' legend_margin('topleft', legend=colnames(a1),
#'     col=1:ncol(a1), lty=1:ncol(a1),
#'     bty='n', cex=.75)
#' 


activationsCueSet <- function(wmlist, cueset, 
	split="_", select.outcomes=NULL, 
	init.value=0, normalize=FALSE){

	# check wmlist:
	if(!is.list(wmlist)){
		if(is.matrix(wmlist)){
			wmlist <- list(wmlist)
		}else{
			stop("Argument wmlist should be list of weightmatrices or a single weightmatrix.")
		}
	}

	# extract all outcomes from wmlist:
	outcomes <- colnames(getWM(wmlist))
	if(!is.null(select.outcomes)){
		outcomes <- select.outcomes
	}
	
	outlist <- list()

	for(i in cueset){
		# calculate raw activations for each of the outcomes:
		out <- lapply(wmlist, function(cur.wm){
			return(activationsMatrix(wm=cur.wm, cues=i, 
				select.outcomes=paste(outcomes, sep=split),
				split=split, init.value=init.value, normalize=normalize))
		})

		out <- as.data.frame(rbindlist(out))
		outlist[[i]] <- out
		out <- NULL
	}
	
	return(outlist)
}

