% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brt_fit.R
\name{brt_fit}
\alias{brt_fit}
\title{Fit boosted regression tree models to species distribution or abundance data.}
\usage{
brt_fit(
  occ.data,
  response.col,
  varnames,
  distribution,
  block.col,
  weights.col,
  test.data,
  interaction.depth,
  n.trees = 5000,
  shrinkage = 0.001
)
}
\arguments{
\item{occ.data}{a data frame, the data to fit boosted regression tree models to, containing
columns for model response and explanatory variable data. If required, \code{occ.data} should contain
\code{block.col} and \code{weights.col} columns too.}

\item{response.col}{a character string, the name of the column in \code{occ.data} containing response
variable column.}

\item{varnames}{a character vector, the names of the columns containing model explanatory
variables in \code{occ.data.}}

\item{distribution}{a character string, the model distribution family to use, such as \code{gaussian},
\code{poisson} or \code{bernoulli}.}

\item{block.col}{optional; a character string, the name of the column in \code{occ.data} containing
spatio-temporal block numbers for \code{occ.data} splitting. See details for more information.}

\item{weights.col}{a character string, the name of the column in \code{occ.data} containing
spatio-temporal sampling effort weights to be used in the model fitting process.}

\item{test.data}{optional; a data frame, the testing dataset for optimising \code{interaction.depth}
when blocking is not used.}

\item{interaction.depth}{optional; an integer specifying the maximum depth of each tree (i.e.
highest level of variable interactions allowed). Default optimises depth between 1 and 4.}

\item{n.trees}{optional; an integer, the number of trees in boosted regression tree models.
Default is 5000.}

\item{shrinkage}{optional; an integer, the shrinkage parameter applied to each tree in the boosted
regression tree expansion. Also known as the learning rate. Default is 0.001.}
}
\value{
Returns a \code{gbm} model object or list of \code{gbm} model objects.
}
\description{
Fit gradient boosting boosted regression tree models to species distribution and abundance data
and associated dynamic explanatory variables.
}
\details{
This function calculates a gradient boosting \code{gbm} object for the response and
explanatory variable data provided, using the \code{gbm} R package (Greenwell et al., 2019).

Key functionality for dynamic SDMs within \code{brt_fit()} includes:
\itemize{
\item Optimise \code{interaction.depth}
}

If \code{interaction.depth} is not given, then \code{brt_fit()} will vary the interaction depth parameter
between 1 (an additive model) and 4 (four-way interaction model). For each \code{interaction.depth}
value, model performance is measured by calculating the root-mean-square error of model
predictions compared to actual values in the testing data. The \code{interaction.depth} value that
results in the lowest root-mean-square error is used when fitting the returned model.

The model testing dataset used can either be given using \code{test.data} or \code{block.col}
(expanded on below).
\itemize{
\item Split by spatio-temporal blocks to account for spatial and temporal autocorrelation
}

If \code{block.col} is specified, then each unique block is excluded in a jack-knife approach
following Bagchi et al., (2013). This approach uses each block as the model testing dataset in
numerical order, whilst all other \code{block.col} blocks are used as training data for the boosted
regression tree model.

In this case, the function returns a list of fitted boosted regression tree models equal to the
length of unique blocking categories in \code{block.col}.

If \code{block.col} is not given, models are fit to all occ.data and a single \code{gbm} model is
returned.
\itemize{
\item Weighted by spatio-temporal sampling effort
}

If \code{weights.col} is specified, records are weighted by their associated value in this column
when model fitting. For instance, the user may wish to down weigh the importance of records
collected at oversampled sites and times when fitting models, and vice versa, to account for
spatio-temporal biases in occurrence records(Stolar and Nielsen, 2015) .
}
\examples{

data("sample_explan_data")
\dontshow{
sample_filt_data<-sample_explan_data[1:65,]
}
split <- sample(c(TRUE, FALSE),
               replace=TRUE,
               nrow(sample_explan_data),
               prob = c(0.75, 0.25))

training <- sample_explan_data[split, ]
testing <- sample_explan_data[!split, ]

brt_fit(
 occ.data = training,
 test.data = testing,
 response.col = "presence.absence",
 distribution = "bernoulli",
 varnames = colnames(training)[14:16],
 interaction.depth = 2
)

}
\references{
Bagchi, R., Crosby, M., Huntley, B., Hole, D. G., Butchart, S. H. M., Collingham, Y.,
Kalra, M., Rajkumar, J., Rahmani, A. & Pandey, M. 2013. Evaluating the effectiveness of
conservation site networks under climate change: accounting for uncertainty. Global Change
Biology, 19, 1236-1248.

Greenwell, B., Boehmke, B., Cunningham, J., & GBM Developers. 2019.
Package ‘gbm’. R package version, 2.

Stolar, J. & Nielsen, S. E. 2015. Accounting For Spatially Biased Sampling Effort In
Presence-Only Species Distribution Modelling. Diversity And Distributions, 21, 595-608.
}
