#' A function to find the maximum and minimum probability of a permutation.
#'
#' @param z vector of doses
#' @param gamma The nonnegative sensitivity parameter; gamma = 0 means no
#' unmeasured confounding.
#'
#' @return a list containing the maximum and minimum probability of a permutation
#' under the Rosenbaum model with doses z and unmeasured confounding level gamma.
#' @export
#'
#' @examples
#' # A vector of observed doses
#' doses <- c(0, 0.1, 0.4, 0.8)
#' bounds <- prob_bounds(z = doses, gamma = 1)
prob_bounds <- function(z, gamma) {
  # number of subjects.
  n <- length(z)
  # sort the doses.
  z <- sort(z)
  # The matrix of permutations of the doses.
  permZ<-unique(gtools::permutations(n,n,z,set = FALSE))

  # functions to maximize
  local_opts <- list( "algorithm" = "NLOPT_LD_MMA",
                      "xtol_rel" = 1.0e-7 )
  opts <- list( "algorithm" = "NLOPT_LN_COBYLA",
                "xtol_rel" = 1.0e-7,
                "maxeval" = 1000,
                "local_opts" = local_opts )
  min_prob <- function(u, z, gamma, permZ) {
    probs <- exp(gamma * (permZ %*% u))
    sum_probs <- sum(probs)
    return(exp(gamma * sum(z * u)) / sum_probs)
  }
  max_prob <- function(u, z, gamma, permZ) {
    probs <- exp(gamma * (permZ %*% u))
    sum_probs <- sum(probs)
    return(-exp(gamma * sum(z * u)) / sum_probs)
  }

  # initialize the optimization
  res_min <- nloptr::nloptr(x0=as.vector(stats::runif(n)),
                     eval_f=min_prob,
                     z=z,gamma=gamma,permZ=permZ,
                     lb=rep(0,n),
                     ub=rep(1,n),
                     opts=opts)
  res_max <- nloptr::nloptr(x0=as.vector(stats::runif(n)),
                    eval_f=max_prob,
                    z=z,gamma=gamma,permZ=permZ,
                    lb=rep(0,n),
                    ub=rep(1,n),
                    opts=opts)

  return(list(min_obj = res_min, max_obj = res_max,
              min_prob = res_min$objective,
              max_prob = -res_max$objective))
}

#' Find the max ratio of probabilities between two permutations.
#'
#' @param z vector of doses
#' @param gamma level of unmeasured confounding
#'
#' @return the maximum ratio between the probability of two different permutations
#' under the Rosenbaum model with doses z and unmeasured confounding level gamma.
#' @export
#'
#' @examples
#' # A vector of observed doses
#' doses <- c(0, 0.1, 0.4, 0.8)
#' ratio <- max_ratio(z = doses, gamma = 1)
max_ratio <- function(z, gamma) {
  z <- sort(z)
  n <- length(z)
  return(exp(gamma*(sum(z[(ceiling(n/2)+1):n])-sum(z[1:floor(n/2)]))))
}

#' Find the max ratio of probabilities between two permutations.
#'
#' @param z vector of doses
#' @param gamma The nonnegative sensitivity parameter; gamma = 0 means no
#' unmeasured confounding.
#'
#' @return the maximum ratio between the probability of two different permutations
#' under the Rosenbaum model with doses z and unmeasured confounding level gamma.
#' @export
#'
#' @examples
#' # A vector of observed doses
#' doses <- c(0, 0.1, 0.4, 0.8)
#' ratio <- max_ratio_new(z = doses, gamma = 1)
max_ratio_new <- function(z, gamma) {
  z <- sort(z)
  n <- length(z)
  objective.in <- z
  const.mat <- matrix(0, nrow = 1 + n, ncol = n)
  const.mat[1,] <- rep(1, n)
  const.dir <- c("=", rep("<=", n))
  const.rhs <- c(n, rep(2, n))
  for (i in 1:n) {
    const.mat[1 + i , i] <- 1
  }
  model <- lpSolve::lp(direction = "max", objective.in = objective.in,
                       const.mat = const.mat, const.dir = const.dir,
                       const.rhs = const.rhs)
  return(exp(gamma*(model$objval-sum(z))))
}

#' A function to compute a conservative upper bound on the worst-case expectation
#' under the sharp null
#'
#' @param z vector of doses of length n
#' @param gamma The nonnegative sensitivity parameter; gamma = 0 means no
#' unmeasured confounding.
#' @param f_pi a vector of length n! that contains the value of the test statistic
#' under each of the n! permutations of z, with order of f_pi determined by first
#' sorting z into increasing order and calling gtools::permutations on z.
#' @param with_variance whether to return the variance along with the worst-case
#' expectation, default is FALSE.
#'
#' @return a list containing the worst-case expectation, and/or variance and the
#' solution to the optimization problem.
#' @export
#'
#' @examples
#' # A vector of observed doses
#' doses <- c(0, 0.1, 0.4, 0.8)
#' # values of test statistic under 4! permutations
#' values <- c(1, 0.5, 0.3, 0.8, 1, 0.7)
#' upper_bound <- max_expectation(z = doses, gamma = 1, f_pi = values)
max_expectation <- function(z, gamma, f_pi, with_variance = FALSE) {
  # number of subjects.
  n <- length(z)
  # sort the doses.
  z <- sort(z)
  # The matrix of permutations of the doses.
  fullZ <- gtools::permutations(n,n,z,set = FALSE)
  permZ <- unique(fullZ)
  indices <- which(!duplicated(fullZ, fromLast = FALSE))
  perm_matrix <- gtools::permutations(n,n)[indices,]

  n_perms <- nrow(permZ)


  # initialize constraint matrix and objective
  # first constraint is the sum of p_pi is 1
  objective.in <- f_pi
  const.mat <- matrix(0, nrow = 1 + n_perms * (n_perms - 1), ncol = n_perms)
  const.mat[1,] <- rep(1, n_perms)
  const.dir <- c("=")
  const.rhs <- c(1)

  # counter
  count <- 2

  # Add the ratio constraints
  for (i in 1:(n_perms-1)) {
    for (j in (i+1):n_perms) {
      # i and j are the indices satisfying i < j
      diff_ind_ij <- which(perm_matrix[i,] != perm_matrix[j,])
      # i - j diff in dose
      diff_dose <- permZ[i,diff_ind_ij] - permZ[j,diff_ind_ij]
      # mult_constant <- max_ratio(z = diff_dose, gamma = gamma)

      # i over j constraint
      const.mat[count,i] <- 1
      const.mat[count,j] <- -exp(gamma * sum(pmax(diff_dose,0)))
      const.dir <- c(const.dir, "<=")
      const.rhs <- c(const.rhs, 0)
      count <- count + 1

      # j over i constraint
      const.mat[count,i] <- -exp(gamma * sum(pmax(-diff_dose,0)))
      const.mat[count,j] <- 1
      const.dir <- c(const.dir, "<=")
      const.rhs <- c(const.rhs, 0)
      count <- count + 1
    }
  }

  model <- lpSolve::lp(direction = "max", objective.in = objective.in,
                       const.mat = const.mat, const.dir = const.dir,
                       const.rhs = const.rhs)

  max_exp <- model$objval
  if (with_variance) {
    variance = sum(f_pi^2 * model$solution) - max_exp^2
    return(list(max_exp = max_exp, variance = variance, solution = model$solution))
  }
  return(max_exp)

}

#' Find the max ratio of probabilities between two permutations for each matched set.
#'
#' @param Z A length N vector of  observed doses.
#' @param index A length N vector of indices indicating matched set membership.
#' @param gamma The nonnegative sensitivity parameter; gamma = 0 means no
#' unmeasured confounding.
#'
#' @return A vector of length equaling the number of unique indices that contains
#' the maximum ratio between any two permutations for each of the matched sets.
#' @export
#'
#' @examples
#' # A vector of observed doses
#' doses <- c(0, 0.1, 0.4, 0.8, 1)
#' matched_set <- c(1, 1, 1, 2, 2)
#' ratios <- max_ratios_summary(Z = doses, index = matched_set, gamma = 1)
max_ratios_summary <- function(Z, index, gamma) {

  # Matched set indices
  match_index = unique(index)

  # number of matched sets
  nostratum <- length(unique(index))

  # vector of max ratios
  max_ratios <- rep(NA, nostratum)

  # compute test stat from each matched set
  for (j in 1:nostratum) {
    # doses in set j
    doses <- Z[which(index == match_index[j])]
    # sort doses
    z <- sort(doses)
    n <- length(z)
    max_ratios[j] = exp(gamma*(sum(z[(ceiling(n/2)+1):n])-sum(z[1:floor(n/2)])))

  }
 return(max_ratios)
}

