% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dists-categorical.R
\name{dodgr_dists_categorical}
\alias{dodgr_dists_categorical}
\title{Cumulative distances along different edge categories}
\usage{
dodgr_dists_categorical(
  graph,
  from = NULL,
  to = NULL,
  proportions_only = FALSE,
  pairwise = FALSE,
  dlimit = NULL,
  heap = "BHeap",
  quiet = TRUE
)
}
\arguments{
\item{graph}{\code{data.frame} or equivalent object representing the network
graph which must have a column named "edge_type" which labels categories of
edge types along which categorical distances are to be aggregated (see
Note).}

\item{from}{Vector or matrix of points \strong{from} which route distances are to
be calculated, specified as one of the following:
\itemize{
\item Single character vector precisely matching node numbers or names
given in \code{graph$from} or \code{graph$to}.
\item Single vector of integer-ish values, in which case these will be
presumed to specify indices into \link{dodgr_vertices}, and NOT to
correspond to values in the 'from' or 'to' columns of the graph. See the
example below for a demonstration.
\item Matrix or equivalent of longitude and latitude coordinates, in which
case these will be matched on to the nearest coordinates of 'from' and 'to'
points in the graph.
}}

\item{to}{Vector or matrix of points \strong{to} which route distances are to be
calculated. If \code{to} is \code{NULL}, pairwise distances will be calculated from
all \code{from} points to all other nodes in \code{graph}. If both \code{from} and \code{to} are
\code{NULL}, pairwise distances are calculated between all nodes in \code{graph}.}

\item{proportions_only}{If \code{FALSE}, return distance matrices for full
distances and for each edge category; if \code{TRUE}, return single vector of
proportional distances, like the \code{summary} function applied to full
results. See Note.}

\item{pairwise}{If \code{TRUE}, calculate distances only between the ordered
pairs of \code{from} and \code{to}. In this case, neither the \code{proportions_only} nor
\code{dlimit} parameters have any effect, and the result is a single matrix with
one row for each pair of \code{from}-\code{to} points, and one column for each
category.}

\item{dlimit}{If no value to \code{to} is given, distances are aggregated from
each \code{from} point out to the specified distance limit (in the same units as
the edge distances of the input graph). \code{dlimit} only has any effect if \code{to}
is not specified, in which case the \code{proportions_only} argument has no
effect.}

\item{heap}{Type of heap to use in priority queue. Options include
Fibonacci Heap (default; \code{FHeap}), Binary Heap (\code{BHeap}),
\verb{Trinomial Heap (}TriHeap\verb{), Extended Trinomial Heap (}TriHeapExt\verb{, and 2-3 Heap (}Heap23`).}

\item{quiet}{If \code{FALSE}, display progress messages on screen.}
}
\value{
If \code{to} is specified, a list of distance matrices of equal dimensions
(length(from), length(to)), the first of which ("distance") holds the final
distances, while the rest are one matrix for each unique value of
"edge_type", holding the distances traversed along those types of edges only.
Otherwise, a single matrix of total distances along all ways from each point
out to the specified value of \code{dlimit}, along with distances along each of
the different kinds of ways specified in the "edge_type" column of the input
graph.
}
\description{
The main \link{dodgr_distances} function calculates distances
between input sets of origin and destination points, and returns a single
matrix of numeric distances. This function aggregates distances along
categories of edges or segments, and returns an overall distance matrix
(identical to the result of \link{dodgr_distances}), along with one
additional matrix for each edge category.

Edges types must be specified in a column of the input graph named
"edge_type". If this has two types of values (for example, "a"
and "b"), then the function will return two additional distance matrices,
one of total lengths of distances between all pairs of points traversed
along edges of the first type, "a", and one of aggregated distances along
edges of type "b".

See the description of \link{dodgr_distances} for details on the expected
format of input graphs.
}
\note{
The "edge_type" column in the graph can contain any kind of discrete or
categorical values, although integer values of 0 are not permissible. \code{NA}
values are ignored. The function requires one full distance
matrix to be stored for each category of "edge_type" (unless
\code{proportions_only = TRUE}). It is wise to keep numbers of discrete types as
low as possible, especially for large distance matrices.

Setting the \code{proportions_only} flag to \code{TRUE} may be advantageous for
large jobs, because this avoids construction of the full matrices. This may
speed up calculations, but perhaps more importantly it may make possible
calculations which would otherwise require distance matrices too large to be
directly stored.

Calculations are not able to be interrupted (for example, by \code{Ctrl-C}),
and can only be stopped by killing the R process.
}
\examples{
# Prepare a graph for categorical routing by including an "edge_type" column
graph <- weight_streetnet (hampi, wt_profile = "foot")
graph <- graph [graph$component == 1, ]
graph$edge_type <- graph$highway
# Define start and end points for categorical distances; using all vertices
# here.
length (unique (graph$edge_type)) # Number of categories
v <- dodgr_vertices (graph)
from <- to <- v$id [1:100]
d <- dodgr_dists_categorical (graph, from, to)
# Internal 'summary' method to summarise results:
summary (d)

class (d)
length (d)
sapply (d, dim)
# 9 distance matrices, all of same dimensions, first of which is standard
# distance matrix
s <- summary (d) # print summary as proportions along each "edge_type"
# or directly calculate proportions only
dodgr_dists_categorical (graph, from, to,
    proportions_only = TRUE
)

# Pairwise distances return single matrix with number of rows equal to 'from'
# / 'to', and number of columns equal to number of edge types plus one for
# total distances.
d <- dodgr_dists_categorical (graph, from, to, pairwise = TRUE)
class (d)
dim (d)

# The 'dlimit' parameter can be used to calculate total distances along each
# category of edges from a set of points out to specified threshold:
dlimit <- 2000 # in metres
d <- dodgr_dists_categorical (graph, from, dlimit = dlimit)
dim (d) # length(from), length(unique(edge_type)) + 1
}
\seealso{
Other distances: 
\code{\link{dodgr_distances}()},
\code{\link{dodgr_dists}()},
\code{\link{dodgr_dists_nearest}()},
\code{\link{dodgr_paths}()},
\code{\link{dodgr_times}()}
}
\concept{distances}
