% Generated by roxygen2: do not edit by hand
% Please edit documentation in
%   R/extract_most_likely_trait_values_for_focal_time.R
\name{extract_most_likely_states_from_densityMaps_for_focal_time}
\alias{extract_most_likely_states_from_densityMaps_for_focal_time}
\title{Extract categorical trait data mapped on a phylogeny at a given time in the past}
\usage{
extract_most_likely_states_from_densityMaps_for_focal_time(
  densityMaps,
  ace = NULL,
  tip_data = NULL,
  focal_time,
  update_densityMaps = FALSE,
  keep_tip_labels = TRUE
)
}
\arguments{
\item{densityMaps}{List of objects of class \code{"densityMap"}, typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}},
that contains a phylogenetic tree and associated posterior probability of being in a given state along branches.
Each object (i.e., \code{densityMap}) corresponds to a state. The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{ace}{(Optional) Numerical matrix that record the posterior probabilities of ancestral states at internal nodes,
obtained with \code{\link[=prepare_trait_data]{prepare_trait_data()}} as output in the \verb{$ace} slot.
Rows are internal nodes_ID. Columns are states. Values are posterior probabilities of each state per node.
Needed to provide accurate estimates of ancestral states.}

\item{tip_data}{(Optional) Named character string vector of tip states.
Names are nodes_ID of the internal nodes. Needed to provide accurate tip values.}

\item{focal_time}{Integer. The time, in terms of time distance from the present,
at which the tree and mapping must be cut. It must be smaller than the root age of the phylogeny.}

\item{update_densityMaps}{Logical. Specify whether the mapped phylogeny (\code{densityMaps})
provided as input should be updated for visualization and returned among the outputs. Default is \code{FALSE}.
The update consists in cutting off branches and mapping that are younger than the \code{focal_time}.}

\item{keep_tip_labels}{Logical. Specify whether terminal branches with a single descendant tip
must retained their initial \code{tip.label} on the updated densityMaps. Default is \code{TRUE}.
Used only if \code{update_map = TRUE}.}
}
\value{
By default, the function returns a list with three elements.
\itemize{
\item \verb{$trait_data} A named character string vector with ML states found along branches overlapping the \code{focal_time}. Names are the tip.label/tipward node ID.
\item \verb{$focal_time} Integer. The time, in terms of time distance from the present, at which the trait data were extracted.
\item \verb{$trait_data_type} Character string. Define the type of trait data as "categorical". Used in downstream analyses to select appropriate statistical processing.
}

If \code{update_densityMaps = TRUE}, the output is a list with four elements: \verb{$trait_data}, \verb{$focal_time}, \verb{$trait_data_type}, and \verb{$densityMaps}.
\itemize{
\item \verb{$densityMaps} A list of objects of class \code{"densityMap"} that contains the updated \code{densityMap} of each state,
with branches and mapping that are younger than the \code{focal_time} cut off.
The function also adds multiple useful sub-elements to the \verb{$densityMaps$tree} elements.
\itemize{
\item \verb{$root_age} Integer. Stores the age of the root of the tree.
\item \verb{$nodes_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_node_ID} to the \code{initial_node_ID}. Each row is a node.
\item \verb{$initial_nodes_ID} Vector of character strings. Provides the initial ID of internal nodes. Used to plot internal node IDs as labels with \code{\link[ape:nodelabels]{ape::nodelabels()}}.
\item \verb{$edges_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_edge_ID} to the \code{initial_edge_ID}. Each row is an edge/branch.
\item \verb{$initial_edges_ID} Vector of character strings. Provides the initial ID of edges/branches. Used to plot edge/branch IDs as labels with \code{\link[ape:nodelabels]{ape::edgelabels()}}.
}
}
}
\description{
Extracts the most likely states found along branches
at a specific time in the past (i.e. the \code{focal_time}).
Optionally, the function can update the mapped phylogeny (\code{densityMaps})
such as branches overlapping the \code{focal_time} are shorten to the \code{focal_time},
and the trait mapping for the cut off branches are removed
by updating the \verb{$tree$maps} and \verb{$tree$mapped.edge} elements.
}
\details{
The mapped phylogeny (\code{densityMaps}) is cut at a specific time in the past
(i.e. the \code{focal_time}) and the current trait values of the overlapping edges/branches are extracted.

----- Extract \code{trait_data} -----

Most likely states are extracted from the posterior probabilities displayed in the \code{densityMaps}.
The state with the highest probability is assigned to each tip and cut branches at \code{focal_time}.

True ML estimates will be used if \code{tip_data} and/or \code{ace} are provided as optional inputs.
In practice the discrepancy is negligible.

----- Update the \code{densityMaps} -----

To obtain updated \code{densityMaps} alongside the trait data, set \code{update_densityMaps = TRUE}.
The update consists in cutting off branches and mapping that are younger than the \code{focal_time}.
\itemize{
\item When a branch with a single descendant tip is cut and \code{keep_tip_labels = TRUE},
the leaf left is labeled with the tip.label of the unique descendant tip.
\item When a branch with a single descendant tip is cut and \code{keep_tip_labels = FALSE},
the leaf left is labeled with the node ID of the unique descendant tip.
\item In all cases, when a branch with multiple descendant tips (i.e., a clade) is cut,
the leaf left is labeled with the node ID of the MRCA of the cut-off clade.
}

The categorical trait mapping in \code{densityMap} (\verb{$tree$maps} and \verb{$tree$mapped.edge}) is updated accordingly by removing mapping associated with the cut off branches.
}
\examples{
# ----- Example 1: Only extent taxa (Ultrametric tree) ----- #

## Load categorical trait data mapped on a phylogeny
data(eel_cat_3lvl_data, package = "deepSTRAPP")

# Explore data
str(eel_cat_3lvl_data, 1)
eel_cat_3lvl_data$densityMaps # Three density maps: one per state

# Set focal time to 10 Mya
focal_time <- 10

\donttest{ # (May take several minutes to run)
## Extract trait data and update densityMaps for the given focal_time

# Extract from the densityMaps
eel_cat_3lvl_data_10My <- extract_most_likely_states_from_densityMaps_for_focal_time(
   densityMaps = eel_cat_3lvl_data$densityMaps,
   # ace = eel_cat_3lvl_data$ace,
   focal_time = focal_time,
   update_densityMaps = TRUE)

## Print trait data
str(eel_cat_3lvl_data_10My, 1)
eel_cat_3lvl_data_10My$trait_data

## Plot density maps as overlay of all state posterior probabilities

# Plot initial density maps with ACE pies
plot_densityMaps_overlay(densityMaps = eel_cat_3lvl_data$densityMaps, fsize = 0.5)
abline(v = max(phytools::nodeHeights(eel_cat_3lvl_data$densityMaps[[1]]$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot updated densityMaps with ACE pies
plot_densityMaps_overlay(eel_cat_3lvl_data_10My$densityMaps, fsize = 0.7) }


# ----- Example 2: Include fossils (Non-ultrametric tree) ----- #
## Test with non-ultrametric trees like mammals in motmot

## Prepare data

# Load mammals phylogeny and data from the R package motmot included within deepSTRAPP
# Data source: Slater, 2013; DOI: 10.1111/2041-210X.12084
data("mammals", package = "deepSTRAPP")

# Obtain mammal tree
mammals_tree <- mammals$mammal.phy
# Convert mass data into categories
mammals_mass <- setNames(object = mammals$mammal.mass$mean,
                         nm = row.names(mammals$mammal.mass))[mammals_tree$tip.label]
mammals_data <- mammals_mass
mammals_data[seq_along(mammals_data)] <- "small"
mammals_data[mammals_mass > 5] <- "medium"
mammals_data[mammals_mass > 10] <- "large"
table(mammals_data)

\donttest{ # (May take several minutes to run)
## Produce densityMaps using stochastic character mapping based on an equal-rates (ER) Mk model
mammals_cat_data <- prepare_trait_data(tip_data = mammals_data, phylo = mammals_tree,
                                       trait_data_type = "categorical",
                                       evolutionary_models = "ER",
                                       nb_simulations = 100,
                                       plot_map = FALSE)

# Set focal time
focal_time <- 80

## Extract trait data and update densityMaps for the given focal_time

# Extract from the densityMaps
mammals_cat_data_80My <- extract_most_likely_states_from_densityMaps_for_focal_time(
    densityMaps = mammals_cat_data$densityMaps,
    focal_time = focal_time,
    update_densityMaps = TRUE)

## Print trait data
str(mammals_cat_data_80My, 1)
mammals_cat_data_80My$trait_data

## Plot density maps as overlay of all state posterior probabilities

# Plot initial density maps with ACE pies
plot_densityMaps_overlay(densityMaps = mammals_cat_data$densityMaps, fsize = 0.7)
abline(v = max(phytools::nodeHeights(mammals_cat_data$densityMaps[[1]]$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot updated densityMaps with ACE pies
plot_densityMaps_overlay(mammals_cat_data_80My$densityMaps, fsize = 0.8) }

}
\seealso{
\code{\link[=cut_phylo_for_focal_time]{cut_phylo_for_focal_time()}} \code{\link[=cut_densityMaps_for_focal_time]{cut_densityMaps_for_focal_time()}}

Associated main function: \code{\link[=extract_most_likely_trait_values_for_focal_time]{extract_most_likely_trait_values_for_focal_time()}}

Sub-functions for other types of trait data:

\code{\link[=extract_most_likely_trait_values_from_contMap_for_focal_time]{extract_most_likely_trait_values_from_contMap_for_focal_time()}}
\code{\link[=extract_most_likely_ranges_from_densityMaps_for_focal_time]{extract_most_likely_ranges_from_densityMaps_for_focal_time()}}
}
\author{
Maël Doré
}
