# cleaR - Utility for 'DIZ' R Package Development
# Copyright (C) 2020-2022 Universitätsklinikum Erlangen, Germany
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

#' @title Clean the console and environment-variables
#'
#' @description Function to clean the local environment.
#'   The call of this function clears the console and the local environment
#'   variables.
#' @param keep_environment (Optional, boolean) If true, the objects from the
#'   environment will not be deleted/emptied.
#' @param keep_console (Optional, boolean) If true, the console will not
#'   be emptied.
#' @param fast (Optional, boolean) If true, `gctorture(TRUE)` will NOT be
#'   applied which will speedup the function call. For more information see
#'   \code{?gctorture}.
#'
#' @return Nothing.
#'
#' @export
#'
clear <- function(keep_environment = FALSE,
                  keep_console = FALSE,
                  fast = FALSE) {
  ## Inspiration: \url{https://stackoverflow.com/questions/62901476}
  if (!fast) {
    gctorture(TRUE)
  }
  if (!keep_console) {
    ## Cleanup the backend in RStudio:
    ## Clears the console (imitates CTR + L)
    cat("\014")
    if (tolower(.Platform$GUI[1]) == "rgui") {
      cat(
        paste0(
          "\n# There is no elegant way to clear the console in the Rgui.",
          " # Alternatively moving the window a few lines up now.\n"
        )
      )
      cat(rep("\n", 100))
    }
  }
  if (!keep_environment) {
    # Clears the Global Environment/variables/data:
    rm(list = ls(all.names = TRUE, envir = sys.frame(-1)),
       envir = sys.frame(-1))
  }
  ## Garbage collector/Clear unused RAM:
  invisible(gc(full = TRUE))
  ## Turn it off (important or it gets very slow):
  if (!fast) {
    gctorture(FALSE)
  }
}
