% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cgmguru-functions-docs.R
\name{detect_hyperglycemic_events}
\alias{detect_hyperglycemic_events}
\title{Detect Hyperglycemic Events}
\usage{
detect_hyperglycemic_events(df,
 reading_minutes = NULL,
 dur_length = 120,
 end_length = 15,
 start_gl = 250,
 end_gl = 180)
}
\arguments{
\item{df}{A dataframe containing continuous glucose monitoring (CGM) data.
Must include columns:
\itemize{
  \item \code{id}: Subject identifier (string or factor)
  \item \code{time}: Time of measurement (POSIXct)
  \item \code{gl}: Glucose value (integer or numeric, mg/dL)
}}

\item{reading_minutes}{Time interval between readings in minutes (optional)}

\item{dur_length}{Minimum duration in minutes for event classification (default: 120)}

\item{end_length}{End length criteria in minutes (default: 15)}

\item{start_gl}{Starting glucose threshold in mg/dL (default: 250)}

\item{end_gl}{Ending glucose threshold in mg/dL (default: 180)}
}
\value{
A list containing:
\itemize{
  \item \code{events_total}: Tibble with summary statistics per subject (id, total_events, avg_ep_per_day)
  \item \code{events_detailed}: Tibble with detailed event information (id, start_time, start_glucose, end_time, end_glucose, start_indices, end_indices)
}
}
\description{
Identifies and segments hyperglycemic events in CGM data based on international consensus 
CGM metrics (Battelino et al., 2023). Supports three event types:
\itemize{
  \item \strong{Level 1}: \eqn{\geq} 15 consecutive min of \eqn{>} 180 mg/dL, ends with \eqn{\geq} 15 consecutive min \eqn{\leq} 180 mg/dL
  \item \strong{Level 2}: \eqn{\geq} 15 consecutive min of \eqn{>} 250 mg/dL, ends with \eqn{\geq} 15 consecutive min \eqn{\leq} 250 mg/dL
  \item \strong{Extended}: \eqn{>} 250 mg/dL lasting \eqn{\geq} 90 cumulative min within a 120-min period, ends when glucose returns to \eqn{\leq} 180 mg/dL 
    for \eqn{\geq} 15 consecutive min after
}
Events are detected when glucose exceeds the start threshold for the minimum duration and ends
when glucose falls below the end threshold for the specified end length.
}
\section{Units and sampling}{

- \code{reading_minutes} can be a scalar (all rows) or a vector per-row.
- If \code{reading_minutes} is \code{NULL}, duration is computed from \code{time} deltas.
}

\examples{
# Load sample data
library(iglu)
data(example_data_5_subject)
data(example_data_hall)

# Level 1: \eqn{\geq} 15 min \eqn{>} 180 mg/dL, 
# ends \eqn{\leq} 180 \eqn{\geq} 15 min
hyper_lv1 <- detect_hyperglycemic_events(
  example_data_5_subject, 
  start_gl = 180, 
  dur_length = 15, 
  end_length = 15, 
  end_gl = 180
)
print(hyper_lv1$events_total)

# Level 2: \eqn{\geq} 15 min \eqn{>} 250 mg/dL, 
# ends \eqn{\leq} 250 \eqn{\geq} 15 min
hyper_lv2 <- detect_hyperglycemic_events(
  example_data_5_subject, 
  start_gl = 250, 
  dur_length = 15, 
  end_length = 15, 
  end_gl = 250
)
print(hyper_lv2$events_total)

# Extended Hyperglycemia (\eqn{>} 250 mg/dL \eqn{\geq} 90 cumulative min within 120-min period,
# ends \eqn{\leq} 180 mg/dL \eqn{\geq} 15 min after)
hyper_extended <- detect_hyperglycemic_events(example_data_5_subject)
print(hyper_extended$events_total)

# Compare event rates across levels
cat("Level 1 events:", sum(hyper_lv1$events_total$total_events), "\n")
cat("Level 2 events:", sum(hyper_lv2$events_total$total_events), "\n")
cat("Extended events:", sum(hyper_extended$events_total$total_events), "\n")

# Analysis on larger dataset with Level 1 criteria
large_hyper <- detect_hyperglycemic_events(example_data_hall, 
                                          start_gl = 180, 
                                          dur_length = 15, 
                                          end_length = 15, 
                                          end_gl = 180)
print(large_hyper$events_total)

# Analysis on larger dataset with Level 2 criteria
large_hyper_lv2 <- detect_hyperglycemic_events(example_data_hall,
                                               start_gl = 250,
                                               dur_length = 15,
                                               end_length = 15,
                                               end_gl = 250)
print(large_hyper_lv2$events_total)

# Analysis on larger dataset with Extended criteria
large_hyper_extended <- detect_hyperglycemic_events(example_data_hall)
print(large_hyper_extended$events_total)

# View detailed events for specific subject
if(nrow(hyper_lv1$events_detailed) > 0) {
  first_subject <- hyper_lv1$events_detailed$id[1]
  subject_events <- hyper_lv1$events_detailed[hyper_lv1$events_detailed$id == first_subject, ]
  head(subject_events)
}
}
\references{
Battelino, T., et al. (2023). Continuous glucose monitoring and metrics for clinical trials: an international consensus statement. The Lancet Diabetes & Endocrinology, 11(1), 42-57.
}
\seealso{
\link{detect_all_events}
}
