% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clean_mixture.R
\name{clean_mixture}
\alias{clean_mixture}
\alias{clean_mixture.default}
\alias{clean_mixture.vpts}
\alias{clean_mixture.vp}
\title{Partition mixtures of animals using assumptions on airspeeds.}
\usage{
clean_mixture(x, ...)

\method{clean_mixture}{default}(
  x,
  slow = 1,
  fast = 8,
  drop_slow_component = TRUE,
  drop_missing = FALSE,
  keep_mixture = FALSE,
  u_wind,
  v_wind,
  u,
  v,
  ...
)

\method{clean_mixture}{vpts}(
  x,
  slow = 1,
  fast = 8,
  drop_slow_component = TRUE,
  drop_missing = FALSE,
  keep_mixture = FALSE,
  u_wind = "u_wind",
  v_wind = "v_wind",
  ...
)

\method{clean_mixture}{vp}(
  x,
  ...,
  slow = 1,
  fast = 8,
  drop_slow_component = TRUE,
  drop_missing = FALSE,
  keep_mixture = FALSE,
  u_wind = "u_wind",
  v_wind = "v_wind"
)
}
\arguments{
\item{x}{a \code{vp} or \code{vpts} object, or a mixture animal density or linear reflectivity eta in cm\eqn{^2}/km\eqn{^3}.}

\item{...}{\code{eta}, \code{u}, \code{v}, \code{u_wind}, \code{v_wind} arguments, taken from object for \code{vp} or \code{vpts} class.}

\item{slow}{the slow component's airspeed in m/s, typically the airspeed of birds.
Either a single number, or (optionally for \code{vpts}) a numeric vector equal in length to the number of profiles,
or a data column name (see Details).}

\item{fast}{the fast component's airspeed in m/s, typically the airspeed of insects.
Either a single number, or (optionally for \code{vpts}) a numeric vector equal in length to the number of profiles,
or a data column name (see Details).}

\item{drop_slow_component}{when TRUE (default) output density, ground speed and
heading for fast component, when FALSE for slow component.}

\item{drop_missing}{Values \code{eta} without an associated ground speed
and wind speed are set to NA when \code{TRUE}, or returned unaltered when \code{FALSE} (default).}

\item{keep_mixture}{When \code{TRUE} store original mixture reflectivity and speeds as
renamed quantities with \code{mixture_} prefix}

\item{u_wind}{the west to east wind component in m/s. In the case of \code{vp} and \code{vpts} objects
the quantity name for the U-component of the wind.}

\item{v_wind}{the south to north wind component in m/s.  In the case of \code{vp} and \code{vpts} objects
the quantity name for the V-component of the wind.}

\item{u}{the mixture's ground speed u component (west to east) in m/s.}

\item{v}{the mixture's ground speed v component (south to north) in m/s.}
}
\value{
a named list with cleaned densities and speeds.
Output differs depending on whether the fast component is retained
(\code{drop_slow_component}=\code{TRUE}, default) or the slow component (\code{drop_slow_component}=\code{FALSE}, default). Output quantities include:
\itemize{
\item \code{eta}: cleaned reflectivity in cm^2/km^3.
only the fast component (default) or the slow component
(when \code{drop_slow_component} is \code{TRUE}).
\item \code{u}: cleaned ground speed component west to east in m/s.
\item \code{v}: cleaned ground speed component south to north in m/s.
\item \code{airspeed}: the airspeed of the selected component in m/s.
\item \code{airspeed_u}: the u-component (west to east) of the airspeed of the retained component in m/s.
\item \code{airspeed_v}: the v-component (south to north) of the airspeed of the retained component in m/s.
\item \code{heading}: the heading of the selected component in degrees clockwise from north.
\item \code{f}: the reflectivity proportion of the slow component (0-1 range), typically the proportion of insects.
}

For \code{vp} and \code{vpts} objects the quantities \code{eta},\code{u},\code{v} will be updated, and other
quantities listed above will be added.
}
\description{
Partition mixtures of birds and insects using assumptions on their respective airspeeds,
following the approach by Shi et al. (2025).
}
\details{
For a detail description of the methodology see Shi et al. (2025).
Most commonly the fast component refers to migrating birds, while
the slow component refers to insects. The slow component is always
oriented in the direction of the wind by definition.
Note that for mixture airspeeds exceeding the airspeed of the fast component,
all reflectivity is assigned to the fast component. Similarly, for mixture
airspeeds below the airspeed of the slow component, all reflectivity
will be assigned to the slow component.
\subsection{How to use this function?}{
\enumerate{
\item To apply this function to \code{vp} or \code{vpts} data altitudinal wind data
needs to be added to the vertical profile data first. This is most easily
accomplished by first converting the objects to a \code{data.frame} with \code{\link[=as.vp]{as.vp()}} or \code{\link[=as.vpts]{as.vpts()}}.
Wind data can then be added as a new columns to the data.frame.
By default the wind data is expected to be named \code{u_wind} for the U component
and \code{v_wind} for the V component of the wind.
Alternatively, arguments \code{u_wind} and \code{v_wind} can be used to specify different names.
\item Realistic assumptions for the expected airspeed for the slow (insect)
and fast (bird) components need to be provided, using arguments \code{slow} and \code{fast}.
See Shi et al. 2025 for recommendations in choosing these values.
The parameter values for \code{fast} and \code{slow} can be specified as follows:
\itemize{
\item as single values applied to all heights and timestamps
\item as a numeric vector of equal length as the number of profiles in the \code{vpts},
allowing the user to specify changes in the parameter over time
\item as the name of a profile data quantity, allowing the user to specify changes in
the parameter over time and/or altitude. Profile quantities are most easily added
by first converting the \code{vpts} object to a data.frame with \code{\link[=as.data.frame.vpts]{as.data.frame.vpts()}},
adding the values, and back-converting with \link{as.vpts}
}
\item Use \code{drop_slow_component} to toggle between retaining the slow or fast component.
When \code{TRUE} the fast (bird) component is retained. When \code{FALSE} the slow (insect)
component is retained. Note that in this case the corrected ground speed direction will be
identical to the wind direction, and the magnitude of the ground speed will be equal
to the wind speed plus the value of \code{slow}, due to the underlying assumption
of wind following by the slow component.
}
}
}
\examples{
# convert profile object to data.frame
df <- as.data.frame(example_vp, suntime=FALSE)
# add wind u and v component wind data
# (here a NW wind identical at all altitudes)
df$u_wind=3
df$v_wind=-3
# convert back to vp object
my_vp <- as.vp(df)
# partition the mixture:
my_vp_clean <- clean_mixture(my_vp)

# drop the slow component (typically insects)
clean_mixture(100,u=-13,v=13,u_wind=-7,v_wind=6, fast=8, slow=1)
# drop the fast component (typically birds)
clean_mixture(100,u=-13,v=13,u_wind=-7,v_wind=6, fast=8, slow=1, drop_slow_component=FALSE)
# keep the original mixture reflectivity and speed components
clean_mixture(100,u=-13,v=13,u_wind=-7,v_wind=6, fast=8, slow=1, keep_mixture=TRUE)
# keep reflectivity unaltered when one of the speed components is not a number:
clean_mixture(100,u=-13,v=13,u_wind=NaN,v_wind=6, fast=8, slow=1)["eta"]
# set reflectivity to NaN when one of the speed components is not a number:
clean_mixture(100,u=-13,v=13,u_wind=NaN,v_wind=6, fast=8, slow=1, drop_missing=TRUE)["eta"]
}
\references{
\itemize{
\item Shi X, Drucker J, Chapman JW, Sanchez Herrera M, Dokter AM
Analysis of mixtures of birds and insects in weather radar data.
Ornithological Applications. 2025 (in press) \doi{10.1093/ornithapp/duaf020}.
\item Nussbaumer R, Schmid B, Bauer S, Liechti F.
A Gaussian mixture model to separate birds and insects in
single-polarization weather radar data. Remote Sensing. 2021 May 19;13(10):1989 \doi{10.3390/rs13101989}.
}
}
