% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/one_way_anova.R
\name{hanova1}
\alias{hanova1}
\title{Posterior sampling for a 1-way hierarchical ANOVA}
\usage{
hanova1(
  n = 1000,
  resp,
  fac,
  ...,
  prior = "default",
  hpars = NULL,
  param = c("trans", "original"),
  init = NULL,
  mu0 = 0,
  sigma0 = Inf,
  nrep = NULL
)
}
\arguments{
\item{n}{A numeric scalar.  The size of posterior sample required.}

\item{resp}{A numeric vector.  Response values.}

\item{fac}{A vector of class \code{\link{factor}} indicating the group from
which the corresponding element of \code{resp} originates.
Must have the same length as \code{resp}.}

\item{...}{Optional further arguments to be passed to
\code{\link[rust]{ru}}.}

\item{prior}{The log-prior for the parameters of the hyperprior
distribution.  If the user wishes to specify their own prior then
\code{prior} must be an object returned from a call to
\code{\link{set_user_prior}}.
Otherwise, \code{prior} is a character scalar giving the name of the
required in-built prior.
If \code{prior} is not supplied then a default prior is used.
See \strong{Details}.}

\item{hpars}{A numeric vector.  Used to set parameters (if any) in
an in-built prior.  If \code{prior = cauchy} then \code{hpars} is
a numeric vector of length 2 giving the respective scale parameters
of the half-Cauchy priors for \eqn{\sigma_\alpha} and \eqn{\sigma}.}

\item{param}{A character scalar.
If \code{param = "trans"} (the default) then the marginal posterior
of hyperparameter vector \eqn{\phi} is reparameterized in terms of
\eqn{log \sigma_\alpha, log \sigma}.
If \code{param = "original"} the original parameterization,
i.e. \eqn{\sigma_\alpha, \sigma} is used.
The former tends to make the optimizations involved in the
ratio-of-uniforms algorithm more stable and to increase the probability
of acceptance, but at the expense of slower function evaluations.}

\item{init}{A numeric vector. Optional initial estimates sent to
\code{\link[rust]{ru}} in the search for the mode of the posterior
density of (perhaps a subset of) the hyperparameter vector \eqn{\phi}.
If an in-built prior is used then \code{ru} is used to sample from the
marginal posterior density of \eqn{(\sigma_\alpha, \sigma)}, so
\code{init} must have length 2.  Otherwise, \code{init} has length
equal to the argument \code{anova_d} supplied to
\code{\link{set_user_prior}}.}

\item{mu0, sigma0}{A numeric scalar.  Mean and standard deviation of a
normal prior for \eqn{\mu}.  Only used if an in-built prior is used
or if \code{anova_d = 2} is supplied in a call to
\code{\link{set_user_prior}} to set a user-defined prior.
The default, \code{sigma0 = Inf}, sets an improper uniform prior
for \eqn{\mu}.}

\item{nrep}{A numeric scalar.  If \code{nrep} is not \code{NULL} then
\code{nrep} gives the number of replications of the original dataset
simulated from the posterior predictive distribution.
Each replication is based on one of the samples from the posterior
distribution.  Therefore, \code{nrep} must not be greater than \code{n}.
In that event \code{nrep} is set equal to \code{n}.}
}
\value{
An object (list) of class \code{"hef"}, which has the same
  structure as an object of class "ru" returned from \code{\link[rust]{ru}}.
  In particular, the columns of the \code{n}-row matrix \code{sim_vals}
  contain the simulated values of \eqn{\phi}.
  In addition this list contains the arguments \code{model}, \code{resp},
  \code{fac} and \code{prior} detailed above and an \code{n} by \eqn{I}
  matrix \code{theta_sim_vals}: column \eqn{i} contains the simulated
  values of  \eqn{\theta}\eqn{i}.  Also included are
  \code{data = cbind(resp, fac)} and \code{summary_stats} a list
  containing: the number of groups \code{I}; the numbers of responses
  each group \code{ni}; the total number of observations; the sample mean
  response in each group; the sum of squared deviations from the
  group means \code{s}; the arguments to \code{hanova1} \code{mu0} and
  \code{sigma0}; \code{call}: the matched call to \code{hanova1}.
}
\description{
Produces random samples from the posterior distribution of the parameters
of a 1-way hierarchical ANOVA model.
}
\details{
Consider \eqn{I} independent experiments in which the \eqn{ni} responses
  \strong{\eqn{y}}\eqn{i} from experiment/group \eqn{i} are normally
  distributed with mean \eqn{\theta i} and standard deviation \eqn{\sigma}.
  The population parameters \eqn{\theta}1, ...,
  \eqn{\theta}\eqn{I} are modelled as random samples from a normal
  distribution with mean \eqn{\mu} and standard deviation
  \eqn{\sigma_\alpha}.  Let \eqn{\phi = (\mu, \sigma_\alpha, \sigma)}.
  Conditionally on \eqn{\theta}1, ..., \eqn{\theta}\eqn{I},
  \strong{\eqn{y}}1, ..., \strong{\eqn{y}}\eqn{I}
  are independent of each other and are independent of \eqn{\phi}.
  A \emph{hyperprior} is placed on \eqn{\phi}.
  The user can either choose parameter values of a default hyperprior or
  specify their own hyperprior using \code{\link{set_user_prior}}.

  The \code{\link[rust]{ru}} function in the \code{\link[rust]{rust}}
  package is used to draw a random sample from the marginal posterior
  of the hyperparameter vector \eqn{\phi}.
  Then, conditional on these values, population parameters are sampled
  directly from the conditional posterior density of
  \eqn{\theta}1, ..., \eqn{\theta}\eqn{I} given \eqn{\phi} and the data.
  See the vignette("bang-c-anova-vignette", package = "bang")
  for details.

  The following priors are specified up to proportionality.

\emph{Priors:}

\code{prior = "bda"} (the default):
\eqn{\pi(\mu, \sigma_\alpha, \sigma) = 1/\sigma,}
that is, a uniform prior for \eqn{(\mu, \sigma_\alpha, log \sigma)},
for \eqn{\sigma_\alpha > 0} and \eqn{\sigma > 0}.
The data must contain at least 3 groups, that is, \code{fac} must have
at least 3 levels, for a proper posterior density to be obtained.
[See Sections 5.7 and  11.6 of Gelman et al. (2014).]

\code{prior = "unif"}:
\eqn{\pi(\mu, \sigma_\alpha, \sigma) = 1,}
that is, a uniform prior for \eqn{(\mu, \sigma_\alpha, \sigma)},
for \eqn{\sigma_\alpha > 0} and \eqn{\sigma > 0}.
[See Section 11.6 of Gelman et al. (2014).]

\code{prior = "cauchy"}: independent half-Cauchy priors for
\eqn{\sigma_\alpha} and \eqn{\sigma} with respective scale parameters
\eqn{A_\alpha} and \eqn{A}, that is,
\eqn{\pi(\sigma_\alpha, \sigma) =
  1 / [(1 + \sigma_\alpha^2 / A_\alpha^2) (1 + \sigma^2 / A^2)].}
[See Gelman (2006).]  The scale parameters (\eqn{A_\alpha}, \eqn{A})
are specified using \code{hpars} = (\eqn{A_\alpha}, \eqn{A}).
The default setting is \code{hpars = c(10, 10).}

\emph{Parameterizations for sampling:}

 \code{param = "original"} is (\eqn{\mu, \sigma_\alpha, \sigma}),
 \code{param = "trans"} (the default) is
  \eqn{\phi1 = \mu, \phi2 = log \sigma_\alpha, \phi3 = log \sigma}.
}
\examples{
# ======= Late 21st Century Global Temperature Data =======

# Extract data for RCP2.6
RCP26_2 <- temp2[temp2$RCP == "rcp26", ]

# Sample from the posterior under the default `noninformative' flat prior
# for (mu, sigma_alpha, log(sigma)).  Ratio-of-uniforms is used to sample
# from the marginal posterior for (log(sigma_alpha), log(sigma)).
temp_res <- hanova1(resp = RCP26_2[, 1], fac = RCP26_2[, 2])

# Plot of sampled values of (sigma_alpha, sigma)
plot(temp_res, params = "ru")

# Plot of sampled values of (log(sigma_alpha), log(sigma))
# (centred at (0,0))
plot(temp_res, ru_scale = TRUE)

# Plot of sampled values of (mu, sigma_alpha, sigma)
plot(temp_res)

# Estimated marginal posterior densities of the mean for each GCM
plot(temp_res, params = "pop", which_pop = "all", one_plot = TRUE)

# Posterior sample quantiles
probs <- c(2.5, 25, 50, 75, 97.5) / 100
round(t(apply(temp_res$sim_vals, 2, quantile, probs = probs)), 2)

# Ratio-of-uniforms information and posterior sample summaries
summary(temp_res)

# ======= Coagulation time data, from Table 11.2 Gelman et al (2014) =======

# With only 4 groups the posterior for sigma_alpha has a heavy right tail if
# the default `noninformative' flat prior for (mu, sigma_alpha, log(sigma))
# is used.  If we try to sample from the marginal posterior for
# (sigma_alpha, sigma) using the default generalized ratio-of-uniforms
# runing parameter value r = 1/2 then the acceptance region is not bounded.

# Two remedies: reparameterize the posterior and/or increase the value of r.

# (log(sigma_alpha), log(sigma)) parameterization, ru parameter r = 1/2
coag1 <- hanova1(resp = coagulation[, 1], fac = coagulation[, 2])

# (sigma_alpha, sigma) parameterization, ru parameter r = 1
coag2 <- hanova1(resp = coagulation[, 1], fac = coagulation[, 2],
               param = "original", r = 1)

# Values to compare to those in Table 11.3 of Gelman et al (2014)
all1 <- cbind(coag1$theta_sim_vals, coag1$sim_vals)
all2 <- cbind(coag2$theta_sim_vals, coag2$sim_vals)
round(t(apply(all1, 2, quantile, probs = probs)), 1)
round(t(apply(all2, 2, quantile, probs = probs)), 1)

# Pairwise plots of posterior samples from the group means
plot(coag1, which_pop = "all", plot_type = "pairs")

# Independent half-Cauchy priors for sigma_alpha and sigma
coag3 <- hanova1(resp = coagulation[, 1], fac = coagulation[, 2],
                 param = "original", prior = "cauchy", hpars = c(10, 1e6))

}
\references{
Gelman, A., Carlin, J. B., Stern, H. S. Dunson, D. B.,
 Vehtari, A. and Rubin, D. B. (2014) \emph{Bayesian Data Analysis}.
 Chapman & Hall / CRC.

Gelman, A. (2006) Prior distributions for variance
  parameters in hierarchical models. \emph{Bayesian Analysis},
  \strong{1}(3), 515-533. \doi{10.1214/06-BA117A}.
}
\seealso{
The \code{\link[rust]{ru}} function in the \code{\link[rust]{rust}}
  package for details of the arguments that can be passed to \code{ru} via
  \code{hanova1}.

\code{\link{hef}} for hierarchical exponential family models.

\code{\link{set_user_prior}} to set a user-defined prior.
}
