% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aoe_sample.R
\name{aoe_sample}
\alias{aoe_sample}
\alias{aoe_sample.default}
\alias{aoe_sample.aoe_result}
\title{Stratified Sampling from AoE Results}
\usage{
aoe_sample(x, ...)

\method{aoe_sample}{default}(x, ...)

\method{aoe_sample}{aoe_result}(
  x,
  n = NULL,
  ratio = c(core = 0.5, halo = 0.5),
  replace = FALSE,
  by = c("overall", "support"),
  ...
)
}
\arguments{
\item{x}{An \code{aoe_result} object returned by \code{\link[=aoe]{aoe()}} or \code{\link[=aoe_expand]{aoe_expand()}}.}

\item{...}{Additional arguments passed to methods.}

\item{n}{Total number of points to sample. If \code{NULL}, uses all available
points subject to the ratio constraint (i.e., downsamples the larger group).}

\item{ratio}{Named numeric vector specifying the target proportion of core
and halo points. Must sum to 1. Default is \code{c(core = 0.5, halo = 0.5)}
for equal representation.}

\item{replace}{Logical. Sample with replacement? Default is \code{FALSE}.
If \code{FALSE} and \code{n} exceeds available points in a stratum, that stratum
contributes all its points.}

\item{by}{Character. Stratification grouping:
\itemize{
\item \code{"overall"} (default): sample from all points regardless of support
\item \code{"support"}: apply ratio within each support separately
}}
}
\value{
An \code{aoe_result} object containing the sampled points, preserving
all original columns and attributes. Has additional attribute
\code{sample_info} with details about the sampling.
}
\description{
Sample points from an \code{aoe_result} with control over core/halo balance.
This is useful when core regions dominate due to point density, and you
want balanced representation for modelling.
}
\details{
\subsection{Sampling modes}{

\strong{Fixed n}: When \code{n} is specified, the function samples exactly \code{n} points
(or fewer if not enough available), distributed according to \code{ratio}.

\strong{Balanced downsampling}: When \code{n} is \code{NULL}, the function downsamples
the larger stratum to match the smaller one according to \code{ratio}.
For example, with ratio \code{c(core = 0.5, halo = 0.5)} and 100 core + 20 halo
points, it returns 20 core + 20 halo = 40 points.
}

\subsection{Multiple supports}{

With \code{by = "support"}, sampling is done independently within each support,
then results are combined. This ensures each support contributes balanced
samples. With \code{by = "overall"}, all points are pooled first.
}
}
\examples{
library(sf)

support <- st_as_sf(
  data.frame(id = 1),
  geometry = st_sfc(st_polygon(list(
    cbind(c(0, 100, 100, 0, 0), c(0, 0, 100, 100, 0))
  ))),
  crs = 32631
)

# Many points in core, few in halo
set.seed(42)
pts <- st_as_sf(
  data.frame(id = 1:60),
  geometry = st_sfc(c(
    lapply(1:50, function(i) st_point(c(runif(1, 10, 90), runif(1, 10, 90)))),
    lapply(1:10, function(i) st_point(c(runif(1, 110, 140), runif(1, 10, 90))))
  )),
  crs = 32631
)

result <- aoe(pts, support, scale = 1)

# Balance core/halo (downsamples core to match halo)
balanced <- aoe_sample(result)

# Fixed sample size with 70/30 split
sampled <- aoe_sample(result, n = 20, ratio = c(core = 0.7, halo = 0.3))

}
\seealso{
\code{\link[=aoe]{aoe()}} for computing AoE classifications
}
