#' Apply covariates to PK(PD) dataset
#'
#' Add covariates to a dataset built by pk_build() or pk_combine()
#' Can add subject-level covariates (by any ID variable) or time-varying (by any time variable)
#'
#' @param df PK(PD) dataframe generated by pk_build
#' @param cov dataframe of covariates
#' @param id.by id variable to merge covariates
#' @param time.by time variable to merge covariates
#' @param direction fill direction for time-varying covariates
#' @param exp treats new covariates as exposure metrics when TRUE
#' @param ebe treats new covariates as empirical bayes estimates when TRUE
#' @param cov.rnd covariate rounding parameter
#' @param na value to replace NA numeric covariates
#' @param demo.map toggle pre-set numeric values for SEX, RACE, and ETHNIC demographic variables
#' @param keep.other filter to keep or remove other events, EVID = 2
#'
#' @return PK(PD) dataset with additional covariates]
#'
#' @examples
#' ## Simple ex domain with 1 subject and 1 dose
#' ex <- data.frame(STUDYID = "ABC101",
#'                  USUBJID = "ABC101-001",
#'                  EXSTDTC = "2000-01-01 10:00:00",
#'                  EXSTDY = 1,
#'                  EXTPTNUM = 0,
#'                  EXDOSE = 100,
#'                  CMT = 1,
#'                  EXTRT = "ABC",
#'                  EXDOSU = "mg",
#'                  VISIT = "Day 1",
#'                  EXTPT = "Dose",
#'                  EXDOSFRQ = "Once",
#'                  EXROUTE = "Oral")
#'
#' ## Simple pc domain with 1 subject and 3 observations
#' pc <- data.frame(USUBJID = "ABC101-001",
#'                  PCDTC = c("2000-01-01 09:40:00",
#'                            "2000-01-01 10:29:00",
#'                            "2000-01-01 12:05:00"),
#'                  PCDY = 1,
#'                  PCTPTNUM = c(0, ##Units of hours
#'                               0.021,
#'                               0.083),
#'                  PCSTRESN = c(NA,
#'                               469,
#'                               870),
#'                  PCLLOQ = 25,
#'                  CMT = 2,
#'                  VISIT = "Day 1",
#'                  PCTPT = c("Pre-dose",
#'                            "30-min post-dose",
#'                            "2-hr post-dose"),
#'                  PCTEST = "ABC",
#'                  PCSTRESU = "ug/mL")
#'
#' ## Create with pk_build()
#' df <- pk_build(ex, pc)
#'
#' ## Simple dm domain for the corresponding study
#' dm <- data.frame(USUBJID = c("ABC101-001",
#'                              "ABC101-002",
#'                              "ABC101-003"),
#'                  AGE = c(45,
#'                          37,
#'                          73),
#'                  AGEU = "years",
#'                  SEX = c("Male",
#'                          "Female",
#'                          "Male"),
#'                  RACE = c("White",
#'                           "White",
#'                           "Black"),
#'                  ETHNIC = c("Not Hispanic/Latino",
#'                             "Not Hispanic/Latino",
#'                             "Not Hispanic/Latino"))
#'
#' ## Add covariates with cov_apply()
#' df1 <- cov_apply(df, dm)
#'
#' @export
#'
cov_apply <- function(df, cov, id.by="USUBJID", time.by=NA,
                      direction="downup", exp = FALSE, ebe = FALSE,
                      cov.rnd=NA, na=-999, demo.map=TRUE, keep.other=TRUE) {
  DTIM <- FDOSE <- .data <- EVID <- ATFD <- USUBJID <- ID <- NULL
  SUBJID <- NSTUDY <- NSTUDYC <- TIMEU <- CMT <- KEEP <- NULL
  ###QC id.by###
  if (length(id.by)>1) { #only one id type
    stop("cov_apply can only fill by one ID type.")
  }
  if (!(as.character(id.by) %in% c("USUBJID", "SUBJID", "ID"))) { #limit id.by inputs
    stop("id.by must be one of the following options: USUBJID, SUBJID, ID")
  }

  ###QC time.by###
  if (length(time.by)>1) {
    stop("cov_apply can only fill by one time type.")
  }
  if (!(is.na(time.by) | as.character(time.by) %in% c(NA, "DTIM", "ATFD", "NTFD", "ATLD", "NTLD", "NTLC", "NDAY"))) {
    stop("time.by must be one of the following options: NA (subject-level attribute), DTIM, ATFD, ATLD, NTFD, NTLC, NTLD, NDAY")
  }

  ###QC direction###
    if (length(direction)>1) {
    stop("cov_apply can only fill in one direction.")
  }

  if (!(direction %in% c("down", "up", "downup", "updown"))) {
    stop("direction must be one of the following (tidy) options: down, up, downup, updown")
  }



  ###QC ebe and exp###
  if (!is.logical(exp)) {
    stop("exp must be TRUE or FALSE")
  }

  if (!is.logical(ebe)) {
    stop("ebe must be TRUE or FALSE")
  }

  if (ebe==TRUE & exp==TRUE) {
    stop("Only one of exp or ebe can be selected as TRUE.")
  }

  ###QC cov.rnd###
  if (!is.na(cov.rnd)) {
    if (!is.numeric(cov.rnd)) {
      stop("cov.rnd parameter must be an integer or NA.")
    }
    if (cov.rnd %% 1 != 0) {
      stop("cov.rnd parameter must be an integer or NA.")
    }
  }

  ###QC na###
  if (!is.na(na)) {
    if (!is.numeric(na)) {
      stop("na parameter must be NA or numeric,")
    }
  }

  ###QC demo map###
  if (is.na(demo.map)) {
    stop("demo.map parameter must be TRUE or FALSE.")
  }
  if (!is.na(demo.map)) {
    if(!is.logical(demo.map)) {
      stop("demo.map parameter must be TRUE or FALSE.")
    }
  }

  ###QC keep.other###
  if (!is.logical(keep.other)) {
    stop("keep.other parameter must be TRUE or FALSE.")
  }

  ###QC df###
  req.cols <- c(id.by) #always required
  if(!is.na(time.by)) {
    req.cols <- c(req.cols, time.by)
  }

  for (i in req.cols) {
    if (!i %in% colnames(df)) { #make sure req.col are in df
      stop(paste(i, "column not in PKPD dataframe. Cannot merge with covariate dataframe."))
    }
  }
  if (!is.na(time.by)) {
    if (time.by=="DTIM") {
      if(!("FDOSE" %in% colnames(df))) {
        stop(paste("Dataframe (df) must contain FDOSE (date/time of first dose) if merging with DTIM."))
      }

      if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}", df$DTIM[!is.na(df$DTIM)])) {
        if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}", df$DTIM[!is.na(df$DTIM)])) {
          if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}", df$DTIM[!is.na(DTIM)])) {
            stop(paste("DTIM in df is not ISO 8601 format."))
          }
        }
      }

      if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}", df$FDOSE[!is.na(df$FDOSE)])) {
        if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}", df$FDOSE[!is.na(df$FDOSE)])) {
          if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}", df$FDOSE[!is.na(FDOSE)])) {
            stop(paste("FDOSE in df is not ISO 8601 format."))
          }
        }
      }

      df <- dplyr::mutate(df,
                          DTIM = dplyr::case_when(grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%dT%H:%M:%S"),
                                                  grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%d %H:%M:%S"),
                                                  grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%dT%H:%M"),
                                                  grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%d %H:%M"),
                                                  grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%d")),
                          FDOSE = dplyr::case_when(grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}:[0-9]{2}", FDOSE) ~ as.POSIXct(FDOSE, tz="UTC", format="%Y-%m-%dT%H:%M:%S"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}:[0-9]{2}", FDOSE) ~ as.POSIXct(FDOSE, tz="UTC", format="%Y-%m-%d %H:%M:%S"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}", FDOSE) ~ as.POSIXct(FDOSE, tz="UTC", format="%Y-%m-%dT%H:%M"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}", FDOSE) ~ as.POSIXct(FDOSE, tz="UTC", format="%Y-%m-%d %H:%M"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}", FDOSE) ~ as.POSIXct(FDOSE, tz="UTC", format="%Y-%m-%d")))
    }
  }

  ###QC cov###
  req.cols <- c(id.by)
  if(!is.na(time.by)) {
    req.cols <- c(req.cols, time.by)
  }

  for (i in req.cols) {
    if (!(i %in% colnames(cov))) {
      stop(paste(i, "column not in covariate dataframe. Cannot merge with PKPD dataframe.")) #make sure req.col is in cov
    }
    if (is.na(time.by) & i %in% c("DTIM", "ATFD", "ATLD", "NTFD", "NTLC", "NTLD", "NDAY")) {
      stop(paste("cov dataset cannot include time variable", i, "while time.by is NA.")) #if time.by is NA, make sure no time variables in cov
    }
  }

  for (i in colnames(cov)) {
    if (i %in% colnames(df) & !(i %in% req.cols)) {
      stop(paste(i, "already exists in PKPD dataframe.")) #make sure covariate column is not already in df
    }
  }

  if (is.na(time.by)) {
    if (nrow(cov)>length(unique(unlist(cov[, id.by])))) {
      stop("Cannot merge cov at subject-level. At least one subject has more than one observation.")
    }
  }

  if (TRUE %in% is.na(df[, id.by])) {
    stop(paste(id.by, "is missing for at least one row."))
  }
  if (length(unique(unlist(df[, id.by]))) > length(unique(unlist(cov[, id.by])))) {
      warning("At least one subject is included in the dataframe (df), but not in covariate dataframe (cov).")
  }


  if (!is.na(time.by)) {
    check <- dplyr::mutate(cov, Check = paste0(.data[[id.by]], .data[[time.by]]))

    if (nrow(check)>length(unique(check$Check))) {
      warning(paste("Some covariates may not be filled. Some rows have duplicate", id.by, "and", time.by, "values."))
    }

    if (time.by=="DTIM") {
      if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}", cov[!is.na(cov$DTIM),"DTIM"])) {
        if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}", cov[!is.na(cov$DTIM),"DTIM"])) {
          if(FALSE %in% grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}", cov[!is.na(cov$DTIM),"DTIM"])) {
            stop(paste("DTIM in", cov, "is not ISO 8601 format."))
          }
        }
      }

      cov <- dplyr::mutate(cov,
                           DTIM = dplyr::case_when(grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%dT%H:%M:%S"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%d %H:%M:%S"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%dT%H:%M"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%d %H:%M"),
                                                   grepl("[0-9]{4}-[0-9]{2}-[0-9]{2}", DTIM) ~ as.POSIXct(DTIM, tz="UTC", format="%Y-%m-%d")))
    }
  }

  ###Covariate Pre-processing###
  cat.cov.n <- c()
  cat.cov.c <- c()
  cont.cov <- c()
  cont.cov.units <- c()
  exp.cov <- c()
  ebe.cov <- c()

  for (i in 1:length(colnames(cov))) {
    name = colnames(cov)[i]
    if (name %in% c(id.by, time.by)) {next}
    else if (exp==TRUE) {
      if (nchar(name)>7) {
        stop(paste(name, "column name in cov must be 7 characters or fewer."))
      }
      exp.cov <- c(exp.cov, paste0("C", name))
      colnames(cov)[i] <- paste0("C", name)
    }
    else if (ebe==TRUE) {
      if (nchar(name)>7) {
        stop(paste(name, "column name in cov must be 7 characters or fewer."))
      }
      ebe.cov <- c(ebe.cov, paste0("I", name))
      colnames(cov)[i] <- paste0("I", name)
    }
    else if (name=="SEX" & demo.map==TRUE) {
      if (is.na(time.by)) {
        nname <- paste0("N", name)
      }
      else {
        nname <- paste0("T", name)
      }
      cov[, nname] <- NA
      cov[grepl("m|male", cov$SEX, ignore.case = TRUE), nname] <- 0
      cov[grepl("f|female", cov$SEX, ignore.case = TRUE), nname] <- 1
      cov[grepl("unk", cov$SEX, ignore.case = TRUE), nname] <- 2
      cov[grepl("other", cov$SEX, ignore.case = TRUE), nname] <- 3
      cat.cov.n <- c(cat.cov.n, nname)
      cat.cov.c <- c(cat.cov.c, paste0(nname, "C"))
      colnames(cov)[i] <- paste0(nname, "C")
      if(length(sort(unique(unlist(cov[, nname]))))!=length(sort(unique(unlist(cov[, paste0(nname, "C")]))))) {
        warning(paste0("At least one ", nname, " failed to map. Consider setting demo.map = FALSE."))
      }
    }
    else if (name=="RACE" & demo.map==TRUE) {
      if (is.na(time.by)) {
        nname <- paste0("N", name)
      }
      else {
        nname <- paste0("T", name)
      }
      cov[, nname] <- NA
      cov[grepl("white|caucasian", cov$RACE, ignore.case = TRUE), nname] <- 1
      cov[grepl("black|african|aa", cov$RACE, ignore.case = TRUE), nname] <- 2
      cov[grepl("asian", cov$RACE, ignore.case = TRUE) & !grepl("caucasian", cov$RACE, ignore.case=TRUE), nname] <- 3
      cov[grepl("alaskan|native", cov$RACE, ignore.case = TRUE), nname] <- 4
      cov[grepl("hawa|pacific|island", cov$RACE, ignore.case = TRUE), nname] <- 5
      cov[grepl("multiple|mul", cov$RACE, ignore.case = TRUE), nname] <- 6
      cov[grepl("other", cov$RACE, ignore.case = TRUE), nname] <- 7
      cov[grepl("unknown", cov$RACE, ignore.case = TRUE), nname] <- 8
      cat.cov.n <- c(cat.cov.n, nname)
      cat.cov.c <- c(cat.cov.c, paste0(nname, "C"))
      colnames(cov)[i] <- paste0(nname, "C")
      if(length(sort(unique(unlist(cov[, nname]))))!=length(sort(unique(unlist(cov[, paste0(nname, "C")]))))) {
        warning(paste0("At least one ", nname, " failed to map. Consider setting demo.map = FALSE."))
      }
    }
    else if (name=="ETHNIC" & demo.map==TRUE) {
      if (is.na(time.by)) {
        nname <- paste0("N", name)
      }
      else {
        nname <- paste0("T", name)
      }
      cov[, nname] <- NA
      cov[grepl("not", cov$ETHNIC, ignore.case = TRUE), nname] <- 0
      cov[grepl("his", cov$ETHNIC, ignore.case = TRUE) & !grepl("not", cov$ETHNIC, ignore.case=TRUE), nname] <- 1
      cov[grepl("unk", cov$ETHNIC, ignore.case = TRUE), nname] <- 2
      cov[grepl("other", cov$ETHNIC, ignore.case = TRUE), nname] <- 3
      cat.cov.n <- c(cat.cov.n, nname)
      cat.cov.c <- c(cat.cov.c, paste0(nname, "C"))
      colnames(cov)[i] <- paste0(nname, "C")
      if(length(sort(unique(unlist(cov[, nname]))))!=length(sort(unique(unlist(cov[, paste0(nname, "C")]))))) {
        warning(paste0("At least one ", nname, " failed to map. Consider setting demo.map = FALSE."))
      }
    }
    else if (is.numeric(unlist(cov[,name]))){
      if (nchar(name)>7) {
        stop(paste(name, "column name in cov must be 7 characters or fewer."))
      }
      if (is.na(time.by)) {
        index_of_unit_column <- grep(paste0(name, "U"), names(cov))
        if (length(index_of_unit_column) != 1) {
          stop(paste("All numerical covariates in cov need units."))
        }
        if(length(sort(unique(unlist(cov[,paste0(name, "U")]))))>1) {
          stop(paste(name, "has more than one unit."))
        }
        baseline_name <- paste0("B", name)
        baseline_name_units <- paste0("B", name, "U")
        colnames(cov)[i] <- baseline_name
        colnames(cov)[index_of_unit_column] <- baseline_name_units
        cont.cov <- append(cont.cov, baseline_name)
        cont.cov.units <- append(cont.cov.units, baseline_name_units)
      }
      else {
        index_of_unit_column <- grep(paste0(name, "U"), names(cov))
        if (length(index_of_unit_column) != 1) {
          stop(paste("All numerical covariates in cov need units."))
        }
        if(length(sort(unique(unlist(cov[,paste0(name, "U")]))))>1) {
          stop(paste(name, "has more than one unit."))
        }
        baseline_name <- paste0("T", name)
        baseline_name_units <- paste0("T", name, "U")
        colnames(cov)[i] <- baseline_name
        colnames(cov)[index_of_unit_column] <- baseline_name_units
        cont.cov <- append(cont.cov, baseline_name)
        cont.cov.units <- append(cont.cov.units, baseline_name_units)
      }
    }
    else if (substr(name, nchar(name), nchar(name))=="U" & gsub("U$", "", name) %in% colnames(cov)) {
      next
    }
    else {
      if (nchar(name)>6) {
        stop(paste(name, "column name in cov must be 6 characters or fewer."))
      }
      if (is.na(time.by)) {
        nname <- paste0("N", name)
      }
      else {
        nname <- paste0("T", name)
      }
      cat.cov.c <- c(cat.cov.c, paste0(nname, "C"))
      cat.cov.n <- c(cat.cov.n, paste0(nname))
      cov[, nname] <- match(unlist(cov[,name]), sort(unique(unlist(cov[,name]))))
      if(length(unique(unlist(cov[, nname])))==2) {
        cov[, nname] <- cov[, nname]-1
      }
      colnames(cov)[i] <- paste0(nname, "C")
    }
  }

  covs <- c(cat.cov.n, cat.cov.c, cont.cov, cont.cov.units, exp.cov, ebe.cov)

  ###FILL###
  if (is.na(time.by)) {
    df <- dplyr::left_join(df, cov, by=id.by)
  }

  else {
    cov <- dplyr::mutate(cov,
                         EVID = 2,
                         DOMAIN = "TVCOV",
                         KEEP = 0,
                         BUILD = Sys.Date())

    if (time.by=="DTIM") {
      if (!"ATFD" %in% colnames(df)) {
        stop(paste0("If merging by DTIM, ATFD must be included in df."))
      }

      df <- dplyr::mutate(df, KEEP = 1)
      df <- dplyr::bind_rows(df, cov)
      df <- dplyr::arrange(df,
                           .data[[id.by]], .data[[time.by]])
      df <- dplyr::group_by(df, .data[[id.by]])
      df <- tidyr::fill(df, FDOSE, .direction="downup")
      df <- dplyr::ungroup(df)
      df <- dplyr::mutate(df,
                          ATFD = ifelse(EVID==2, as.numeric(difftime(DTIM, FDOSE, units=sort(unique(df$TIMEU)))), ATFD),
                          ATFD = dplyr::case_when(EVID==2 & ATFD%%1==1 ~ ATFD-1,
                                                  EVID==2 & ATFD%%0.1==0.1 ~ ATFD-0.1,
                                                  EVID==2 & ATFD%%0.01==0.01 ~ ATFD-0.01,
                                                  EVID==2 & ATFD%%0.001==0.001 ~ ATFD-0.001,
                                                  EVID==2 ~ ATFD-0.0001,
                                                  TRUE ~ ATFD))
      df <- dplyr::arrange(df, USUBJID, ATFD)
      df <- dplyr::group_by(df, USUBJID)
      df <- tidyr::fill(df, ID, SUBJID, NSTUDY, NSTUDYC, TIMEU,
                        tidyselect::all_of(covs), .direction=direction)
      df <- dplyr::ungroup(df)
      df <- dplyr::arrange(df, ID, ATFD, CMT, EVID)
    }

    else {
      df <- dplyr::mutate(df, KEEP = 1)
      df <- dplyr::bind_rows(df, cov)
      df <- dplyr::arrange(df,
                           .data[[id.by]], .data[[time.by]], -EVID)
      df <- dplyr::group_by(df, .data[[id.by]]) #group
      df <- tidyr::fill(df, ID, SUBJID, NSTUDY, NSTUDYC, TIMEU,
                        tidyselect::all_of(covs), .direction=direction)
      df <- dplyr::ungroup(df)
      df <- dplyr::mutate(df,
                          ATFD = dplyr::case_when(KEEP==0 & time.by=="NTFD" ~ NTFD,
                                                  KEEP==0 & time.by=="NDAY" & TIMEU=="days" ~ NDAY-1,
                                                  KEEP==0 & time.by=="NDAY" & TIMEU=="hours" ~ (24*NDAY-1),
                                                  TRUE ~ ATFD))
      df <- dplyr::arrange(df, ID, ATFD, CMT, EVID)
    }
  }

  ###Round covariates###
  if(is.numeric(cov.rnd)) {
    for (i in c(cat.cov.n, cont.cov, ebe.cov, exp.cov)) {
      cov.num <- round(df[, i], cov.rnd)
      df <- dplyr::select(df, -tidyselect::all_of(i))
      df <- dplyr::bind_cols(df, cov.num)
    }
  }

  ###Fill NA values###
  for (i in c(cat.cov.n, cont.cov, ebe.cov, exp.cov)) {
    df[is.na(df[, i]) | df[, i]==-999, i] <- na
  }

  ###Column order###
  df.cat.cov.n <- cov_find(df, cov="categorical", type="numeric")
  df.cat.cov.c <- cov_find(df, cov="categorical", type="character")

  if(!is.null(cat.cov.n)) {
    if (!is.null(df.cat.cov.n)) {
      df <- dplyr::relocate(df,
                            tidyselect::all_of(cat.cov.n), .after=df.cat.cov.n[length(df.cat.cov.n)])
    }
    else {
      df <- dplyr::relocate(df,
                            tidyselect::all_of(cat.cov.n), .after="DOSEA")
    }
  }
  if(!is.null(cat.cov.c)) {
    if (!is.null(df.cat.cov.c)) {
      df <- dplyr::relocate(df,
                            tidyselect::all_of(cat.cov.c), .after=df.cat.cov.c[length(df.cat.cov.c)])
    }
    else {
      df <- dplyr::relocate(df,
                            tidyselect::all_of(cat.cov.c), .after="DOSEA")
    }
  }
  if(!is.null(cont.cov)) {
    df <- dplyr::relocate(df,
                          tidyselect::all_of(cont.cov), .before="PDOSEF")
  }
  if(!is.null(exp.cov)) {
    df <- dplyr::relocate(df,
                          tidyselect::all_of(exp.cov), .before="PDOSEF")
  }
  if(!is.null(ebe.cov)) {
    df <- dplyr::relocate(df,
                          tidyselect::all_of(ebe.cov), .before="PDOSEF")
  }
  if(!is.null(cont.cov.units)) {
    df <- dplyr::relocate(df,
                          tidyselect::all_of(cont.cov.units), .before="DTIM")
  }

  ###Final filter###
  if (is.na(time.by)) {
  }

  else if(keep.other==FALSE) {
    df <- dplyr::filter(df, KEEP == 1)
    df <- dplyr::select(df, -KEEP)
  }

  else if (time.by %in% c("ATLD", "NTLC", "NTLD")) {
    df <- dplyr::filter(df, KEEP == 1)
    df <- dplyr::select(df, -KEEP)
  }

  else {
    df <- dplyr::select(df, -KEEP)
    df <- dplyr::mutate(df, LINE = dplyr::row_number())
  }

  ###Read out updated dataset###
  return(df)
}
