% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ah2phase.R
\name{ah.2ph}
\alias{ah.2ph}
\title{Fit Additive Hazards Regression Models to Two-phase Sampling}
\usage{
ah.2ph(formula, data, R, Pi, ties, robust = FALSE,
  calibration.variables = NULL, ...)
}
\arguments{
\item{formula}{a formula object for the regression model of the form
response ~ predictors. The outcome is a survival object created by \code{\link[survival]{Surv}}.}

\item{data}{a data frame. Input dataset.}

\item{R}{a phase II membership indicator. A vector of values of 0 and 1.
The subject is selected to phase II if R = 1.}

\item{Pi}{the  probability of a subject to be selected to the phase II subsample.}

\item{ties}{a logical variable. FALSE if there are no ties in the censored failure times.}

\item{robust}{a logical variable.  Robust standard errors are provided if robust = TRUE.}

\item{calibration.variables}{a vector of some column names of the data.
These are the  variables available for every observation. They are used to
calibrate the weight assigned to each subject in order to improve estimation efficiency.}

\item{...}{additional arguments to be passed to the low level regression fitting
functions.}
}
\value{
An object of class 'ah.2h' representing the fit.
}
\description{
The function fits a semiparametric additive hazards model
 \deqn{ \lambda(t|Z=z) = \lambda_0(t) + \beta'z.} to two-phase sampling data.
 The estimating procedures follow Hu (2014).
}
\note{
This function estimates both model-based and robust standard errors. It can be
used to analyze case-cohort studies. It allows subsampling among cases. It can
incorporate the calibration procedure and analyze the combined dataset of phase
I and phase II samples.
}
\examples{
library(survival)
### fit an additive hazards model to two-phase sampling data without calibration
nwts2ph$trel <- nwts2ph$trel + runif(dim(nwts2ph)[1],0,1)*1e-8
fit1 <- ah.2ph(Surv(trel,relaps) ~ age + histol, ties = FALSE, data = nwts2ph, R = in.ph2, Pi = Pi,
 robust = FALSE,  calibration.variables = NULL)
summary(fit1)

### fit an additve hazards model with calibration on age
fit2 <- ah.2ph(Surv(trel,relaps) ~ age + histol, ties = FALSE, data = nwts2ph, R = in.ph2, Pi = Pi,
 robust = FALSE, calibration.variables = "age")
summary(fit2)

### calibrate on age square
### note if users create a  calibration variable, then
### the new variable should be added to the original data frame
nwts2ph$age2 <- nwts2ph$age^2
fit3 <- ah.2ph(Surv(trel,relaps) ~ age + histol, ties = FALSE, data = nwts2ph, R = in.ph2, Pi = Pi,
                                   robust = FALSE, calibration.variables = "age2")
summary(fit3)

#############################################################################
### When phase II samples are obtained by finite Sampling       #############
#############################################################################

### calculating the sample size for each straum
### calculate the strata size
strt.size <- table(nwts2ph$strt)
ph2.strt.size <- table(subset(nwts2ph, in.ph2 == 1)$strt)
### fit an additve hazards model with finite stratified sampling
### calculate the sampling fractions
frac <- ph2.strt.size/strt.size
### treating the problem as bernoulli sampling coupled with calibration on strata sizes
### using frac as the sampling probilities
nwts2ph_by_FPSS <- nwts2ph
nwts2ph_by_FPSS$Pi <- NULL
for (i in 1:length(strt.size)){
  nwts2ph_by_FPSS$Pi[nwts2ph_by_FPSS$strt ==i] <- frac[i]
}

### create strt indicators, which become our calibration variables
for (i in 1:length(strt.size)){
   nwts2ph_by_FPSS$strt_ind <- as.numeric(nwts2ph_by_FPSS$strt ==i)
   names(nwts2ph_by_FPSS)[ncol(nwts2ph_by_FPSS)]= paste0("strt", i)
}
### fit an additve hazards model with finate sampling
fit4 <- ah.2ph(Surv(trel,relaps) ~ age + histol,
                                   data = nwts2ph_by_FPSS, ties = FALSE,
                                   R = in.ph2, Pi = Pi,
                                   robust = FALSE,
                                   calibration.variables = c("strt1","strt2","strt3"))
summary(fit4)
}
\references{
Jie Hu (2014) A Z-estimation System for Two-phase Sampling with Applications to
Additive Hazards Models and  Epidemiologic Studies. Dissertation, University of Washington.
}
\seealso{
\code{\link{predict.ah.2ph}} for prediction based on fitted additive
hazards model with two-phase sampling and \code{\link{nwtsco}} for the description
of nwtsco dataset.
}

