% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect-ac.R
\name{detect_ac}
\alias{detect_ac}
\title{Detect answer copying}
\usage{
detect_ac(
  method,
  psi,
  xi = NULL,
  x = NULL,
  r = NULL,
  interval = c(-4, 4),
  alpha = 0.05
)
}
\arguments{
\item{method}{The answer copying statistic(s) to compute. Options for
score-based statistics are:
\itemize{
\item \code{"OMG_S"} for the conditional \eqn{\omega} statistic (Wollack, 1997).
\item \code{"GBT_S"} for the conditional \eqn{GBT} statistic (van der Linden &
Sotaridona, 2006).
}

Options for response-based statistics are:
\itemize{
\item \code{"OMG_R"} for the conditional \eqn{\omega} statistic (Wollack, 1997).
\item \code{"GBT_R"} for the conditional \eqn{GBT} statistic (van der Linden &
Sotaridona, 2006).
}}

\item{psi}{A matrix of item parameters. Rows correspond to items and columns
to parameters.}

\item{xi}{A matrix of person parameters. Rows correspond to persons and
columns to parameters. If \code{NULL} (default), person parameters are estimated
using maximum likelihood estimation.}

\item{x, r}{Matrices of raw data. Rows correspond to persons and columns to
items. \code{x} is for the item scores and \code{r} the item responses.}

\item{interval}{The interval to search for the person parameters. Default is
\code{c(-4, 4)}.}

\item{alpha}{Value(s) between 0 and 1 indicating the significance level(s)
used for flagging. Default is \code{0.05}.}
}
\value{
A list is returned with the following elements:
\item{stat}{A matrix of answer copying statistics.}
\item{pval}{A matrix of \emph{p}-values.}
\item{flag}{An array of flagging results. The first dimension corresponds to
source-copier pairs, the second dimension to methods, and the third
dimension to significance levels.}
}
\description{
Detect answer copying for all possible source-copier pairs.
}
\examples{
# Setup for Examples 1 and 2 ------------------------------------------------

# Settings
set.seed(0)     # seed for reproducibility
N <- 50         # number of persons
n <- 40         # number of items

# Randomly select 10\% sources and 10\% copiers
s <- sample(1:N, size = N * 0.10)
c <- sample(setdiff(1:N, s), size = N * 0.10)

# Create vector of indicators (1 = copying pair, 0 = non-copying pair)
pair <- t(combn(N, 2))
pair <- rbind(pair, pair[, 2:1])
ind <- ifelse(1:nrow(pair) \%in\% apply(
  rbind(cbind(s, c), cbind(c, s)), 1, function(p)
  which(pair[, 1] == p[1] & pair[, 2] == p[2])), 1, 0)
names(ind) <- paste(pair[, 1], pair[, 2], sep = "-")

# Example 1: Item Scores ----------------------------------------------------

# Generate person parameters for the 3PL model
xi <- cbind(theta = rnorm(N, mean = 0.00, sd = 1.00))

# Generate item parameters for the 3PL model
psi <- cbind(
  a = rlnorm(n, meanlog = 0.00, sdlog = 0.25),
  b = rnorm(n, mean = 0.00, sd = 1.00),
  c = runif(n, min = 0.05, max = 0.30)
)

# Simulate uncontaminated data
x <- sim(psi, xi)$x

# Modify contaminated data by replacing 40\% of the copier scores with source
# scores
for (v in 1:length(c)) {
  ci <- sample(1:n, size = n * 0.40)
  x[c[v], ci] <- x[s[v], ci]
}

# Detect answer copying
out <- detect_ac(
  method = c("OMG_S", "GBT_S"),
  psi = psi,
  x = x
)

# Example 2: Item Responses -------------------------------------------------

# Generate person parameters for the nominal response model
xi <- cbind(eta = rnorm(N, mean = 0.00, sd = 1.00))

# Generate item parameters for the nominal response model
psi <- cbind(
  lambda1 = rnorm(n, mean = -0.50, sd = 0.50),
  lambda2 = rnorm(n, mean = -0.50, sd = 0.50),
  lambda3 = rnorm(n, mean = -0.50, sd = 0.50),
  lambda4 = rnorm(n, mean = 1.50, sd = 0.50),
  zeta1 = rnorm(n, mean = -0.50, sd = 0.50),
  zeta2 = rnorm(n, mean = -0.50, sd = 0.50),
  zeta3 = rnorm(n, mean = -0.50, sd = 0.50),
  zeta4 = rnorm(n, mean = 1.50, sd = 0.50)
)

# Simulate uncontaminated data
r <- sim(psi, xi)$r

# Modify contaminated data by replacing 40\% of the copier responses with
# source responses
for (v in 1:length(c)) {
  ci <- sample(1:n, size = n * 0.40)
  r[c[v], ci] <- r[s[v], ci]
}

# Detect answer copying
out <- detect_ac(
  method = c("OMG_R", "GBT_R"),
  psi = psi,
  r = r
)
}
\references{
van der Linden, W. J., & Sotaridona, L. (2006). Detecting answer copying when
the regular response process follows a known response model. \emph{Journal of
Educational and Behavioral Statistics}, \emph{31}(3), 283--304.

Wollack, J. A. (1997). A nominal response model approach for detecting answer
copying. \emph{Applied Psychological Measurement}, \emph{21}(4), 307--320.
}
\seealso{
\code{\link[=detect_as]{detect_as()}} to detect answer similarity.
}
