# telescoped.upsilon.test.R
#
# Author: Xuye Luo, Joe Song
#
# Updated:
# December 20, 2025. 
#   - Updated the documentation
#
# December 11, 2025

#' @title Find Best Upsilon Test Result in Telescoped Data
#'
#' @description Performs Upsilon tests on telescoped data at multiple resolution levels 
#' (or a specific focus level) to identify the strongest association.
#' The "best" result is determined either by the largest effect size (default) or 
#' the most significant p-value, depending on the \code{esize} parameter.
#'
#' @param x A numeric vector without NA values, same length as y.
#' @param y A numeric vector without NA values, same length as x.
#' @param focus An integer specifying a specific resolution level to test. 
#'              If \code{NULL} (default), tests are performed across all available levels.
#' @param base Integer. The base for exponential scaling in telescoping. Defaults to 2.
#' @param log.p a logical. If \code{TRUE}, 
#'   the \emph{p}-value is calculated in
#'   closed form to \strong{natural logarithm} of \emph{p}-value 
#'   to improve numerical precision when
#'   \emph{p}-value approaches zero.
#'   Defaults to \code{FALSE}.
#' @param esize Logical. If \code{TRUE} (default), sorts results primarily by Effect Size (descending). 
#'              If \code{FALSE}, sorts primarily by P-value (significance) in descending order.
#'
#' @return A data frame containing the best result with columns:
#' \item{p.value}{The p-value (or log p-value) of the Upsilon test.}
#' \item{estimate}{The effect size.}
#' \item{focus}{The resolution level (focus) where this result was found.}
#'
#'
#' @examples
#' library("Upsilon")
#' set.seed(123)
#' n <- 100
#' x <- rnorm(n)
#' y <- rnorm(n)
#' 
#' # Find best result based on Significance (P-value)
#' telescoped.upsilon.test(x, y)
#' @keywords internal
#' export
#' @noRd
telescoped.upsilon.test <- function(
    x, 
    y, 
    focus = NULL, 
    base = 2, 
    log.p = TRUE, 
    esize = FALSE) 
{
  
  # Get telescoped data
  telescope_list <- telescoped.data(x, y, focus = focus, base = base)
  
  # Search across multiple levels
  if (is.list(telescope_list) && is.null(focus)) {
    
    if (length(telescope_list) > 1) {
      telescope_list <- telescope_list[-1]
    }
    
    # Iterate through each level
    upsilon_result_list <- lapply(telescope_list, function(mat) {
      # Use matrix indexing instead of unlist for safety
      val_x <- mat[, 1]
      val_y <- mat[, 2]
      
      results <- fast.upsilon.test(val_x, val_y, log.p = log.p)
      
      data.frame(
        p.value = results$p.value, 
        estimate = results$estimate
      )
    })
    
    upsilon_result <- do.call(rbind, upsilon_result_list)
    
    # Assign focus levels (0 to N-1 based on list index)
    upsilon_result$focus <- seq_len(nrow(upsilon_result)) - 1
    
    # Sorting Logic
    if (esize) {
      ord <- order(
        upsilon_result$estimate, 
        -upsilon_result$p.value, 
        -upsilon_result$focus, 
        decreasing = TRUE
      )
    } else {
      ord <- order(
        -upsilon_result$p.value, 
        -upsilon_result$focus, 
        decreasing = TRUE
      )
    }
    
    upsilon_result_sorted <- upsilon_result[ord, , drop = FALSE]
    upsilon_result_best <- upsilon_result_sorted[1, , drop = FALSE]
    
  } else {
    mat <- if (is.list(telescope_list)) telescope_list[[1]] else telescope_list
    
    results <- fast.upsilon.test(mat[, 1], mat[, 2], log.p = log.p)
    
    upsilon_result_best <- data.frame(
      p.value = results$p.value,
      estimate = results$estimate,
      focus = ifelse(is.null(focus), -1, focus)
    )
  }
  
  return(upsilon_result_best)
}